#pragma once

#include "errors.h"

#include <yplatform/task_context.h>
#include <yplatform/time_traits.h>
#include <yplatform/util/safe_call.h>
#include <yplatform/util/tuple_unpack.h>

#include <functional>
#include <memory>
#include <string>
#include <vector>

namespace collectors {

namespace ph = std::placeholders;

// XXX using context = yplatform::task_context;
using context_ptr = yplatform::task_context_ptr;
namespace time_traits = yplatform::time_traits;

// XXX using std::string
// XXX using std::time_t
using string_opt = std::optional<std::string>;

using shard_id = std::string;
using stid = std::string;
using uid = std::string;
using fid = std::string;
using mid = std::string;
using lid = std::string;
using collector_id = int32_t;

constexpr auto EMPTY_ID = "0";

template <typename Id>
inline bool is_empty_id(const Id& id)
{
    return id.empty() || id == EMPTY_ID;
}

template <typename Id>
inline Id normalize_id(const Id& id)
{
    return id.empty() ? Id(EMPTY_ID) : id;
}

struct global_collector_id
{
    global_collector_id() = default;
    global_collector_id(const collectors::uid& uid, const collectors::collector_id& collector_id);
    global_collector_id(const std::string& str);

    collectors::uid uid;
    collectors::collector_id collector_id = 0;

    std::string to_string() const;
    bool operator<(const global_collector_id& s) const;
};

struct collector_logging_info
{
    uid dst_uid;
    uid src_uid;
    global_collector_id id;

    std::string to_string() const;
};

using uids = std::vector<uid>;
using mids = std::vector<mid>;
using lids = std::vector<lid>;
using collector_ids = std::vector<collector_id>;

using no_data_cb = std::function<void(error)>;
using fid_cb = std::function<void(error, const fid&)>;
using mid_cb = std::function<void(error, const mid&)>;
using collector_ids_cb = std::function<void(error, const collector_ids&)>;
using uids_cb = std::function<void(error, const uids&)>;

template <typename... Cbs>
class multi_cb
{
public:
    multi_cb(const Cbs&... cbs) : cbs(cbs...)
    {
    }

    template <typename... Args>
    void operator()(Args&&... args)
    {
        std::apply(
            [args = std::tuple{ std::forward<Args>(args)... }, this](auto&&... cbs) {
                // Avoiding "this->" causes compiler error, because of "unused this"
                (this->safe_apply(cbs, args), ...);
            },
            cbs);
    }

private:
    // Cannot use in-place implementation because of clang crash
    template <typename F, typename Tuple>
    void safe_apply(F&& f, Tuple&& t)
    {
        // Avoiding std::bind causes compile error, bug?
        yplatform::safe_call(
            std::bind(&std::apply<F, Tuple>, std::forward<F>(f), std::forward<Tuple>(t)));
    }

    std::tuple<Cbs...> cbs;
};

inline bool is_old_popid(const std::string& popid)
{
    for (auto letter : popid)
    {
        if (!std::isdigit(letter))
        {
            return false;
        }
    }
    return popid.size();
}

}
