#include "mailbox.h"

#include <common/msgpack.h>

#include <ymod_httpclient/call.h>
#include <ymod_httpclient/request.h>
#include <ymod_mdb_sharder/users_distributor.h>
#include <ymod_tvm/module.h>
#include <yplatform/coroutine.h>
#include <yplatform/find.h>

#include <memory>

#include <yplatform/yield.h>

namespace collectors::mailbox {

class http_mailbox;
using http_mailbox_ptr = std::shared_ptr<http_mailbox>;

template <typename Callback>
struct call_internal_api_op
{
    using yield_context = yplatform::yield_context<call_internal_api_op>;

    call_internal_api_op(
        context_ptr ctx,
        const yhttp::request& req,
        http_mailbox_ptr mailbox,
        const Callback& cb)
        : ctx(ctx)
        , orig_url(req.url)
        , req(req)
        , mailbox(mailbox)
        , client(yplatform::find<yhttp::call>("http_client"))
        , cb(cb)
    {
    }

    void operator()(yield_context yield_ctx);

    void operator()(typename yield_context::exception_type exception);

    void complete();

    context_ptr ctx;
    std::string orig_url;
    yhttp::request req;
    http_mailbox_ptr mailbox;
    boost::shared_ptr<yhttp::call> client;
    std::decay_t<Callback> cb;
    yhttp::response resp;
    error ec;
};

class http_mailbox
    : public std::enable_shared_from_this<http_mailbox>
    , public read_only_mailbox
{ // XXX order
public:
    http_mailbox(
        context_ptr ctx,
        const uid& uid,
        const std::string& owner_host,
        const std::string& service_ticket,
        settings_ptr settings)
        : ctx_(ctx)
        , uid_(uid)
        , owner_host_(owner_host)
        , ticket_header_("X-Ya-Service-Ticket: " + service_ticket + "\r\n")
        , settings_(settings)
    {
    }

    void get_folders(const folders_cb& cb) override
    {
        if (ctx_->is_cancelled())
            return cb(code::cancelled_context, {}); // XXX del, check in high-level funcs

        auto self = shared_from_this();
        auto internal_cb = [cb, this, self](error ec, const yhttp::response& resp) {
            if (ec)
            {
                TASK_LOG(ctx_, error) << "get_folders error: " << ec.message();
                return cb(code::mailbox_access_error, {});
            }

            if (resp.status != 200)
            {
                TASK_LOG(ctx_, error) << "get_folders error: " << resp.body;
                return cb(code::mailbox_access_error, {});
            }

            cb(ec, msgpack::unpack<folders>(resp.body));
        };
        auto op = std::make_shared<call_internal_api_op<decltype(internal_cb)>>(
            ctx_, make_folders_request(), self, internal_cb);
        yplatform::spawn(op);
    }

    void get_labels(const labels_cb& cb) override
    {
        if (ctx_->is_cancelled())
            return cb(code::cancelled_context, {}); // XXX del, check in high-level funcs

        auto self = shared_from_this();
        auto internal_cb = [cb, this, self](error ec, const yhttp::response& resp) {
            if (ec)
            {
                TASK_LOG(ctx_, error) << "get_labels error: " << ec.message();
                return cb(code::mailbox_access_error, {});
            }

            if (resp.status != 200)
            {
                TASK_LOG(ctx_, error) << "get_labels error: " << resp.body;
                return cb(code::mailbox_access_error, {});
            }

            cb(ec, msgpack::unpack<labels>(resp.body));
        };
        auto op = std::make_shared<call_internal_api_op<decltype(internal_cb)>>(
            ctx_, make_labels_request(), self, internal_cb);
        yplatform::spawn(op);
    }

    void get_next_message_chunk(const mid& mid, uint64_t count, const messages_cb& cb) override
    {
        if (ctx_->is_cancelled())
            return cb(code::cancelled_context, {}); // XXX del, check in high-level funcs

        auto self = shared_from_this();
        auto internal_cb = [cb, this, self](error ec, const yhttp::response& resp) {
            if (ec)
            {
                TASK_LOG(ctx_, error) << "get_next_message_chunk error: " << ec.message();
                return cb(code::mailbox_access_error, {});
            }

            if (resp.status != 200)
            {
                TASK_LOG(ctx_, error) << "get_next_message_chunk error: " << resp.body;
                return cb(code::mailbox_access_error, {});
            }

            cb(ec, msgpack::unpack<messages>(resp.body));
        };
        auto op = std::make_shared<call_internal_api_op<decltype(internal_cb)>>(
            ctx_, make_next_message_chunk_request(mid, count), self, internal_cb);
        yplatform::spawn(op);
    }

    template <typename Callback>
    void refresh_owner(Callback&& cb)
    {
        auto distributor =
            yplatform::find<ymod_mdb_sharder::users_distributor>("users_distributor");
        auto self = shared_from_this();
        distributor->get_owner(
            ctx_,
            std::stoull(uid_),
            [cb, this, self](error ec, const ymod_mdb_sharder::node_info& node) mutable {
                if (!ec)
                {
                    owner_host_ = node.host;
                }
                cb(ec);
            });
    }

    std::string get_owner_address()
    {
        return owner_host_ + ":" + std::to_string(settings_->internal_api_port);
    }

protected:
    virtual yhttp::request make_folders_request()
    {
        auto params = yhttp::url_encode({ { "uid", uid_ }, { "_", ctx_->uniq_id() } });
        return yhttp::request::GET("/folders" + params, ticket_header_);
    }

    virtual yhttp::request make_labels_request()
    {
        auto params = yhttp::url_encode({ { "uid", uid_ }, { "_", ctx_->uniq_id() } });
        return yhttp::request::GET("/labels" + params, ticket_header_);
    }

    virtual yhttp::request make_next_message_chunk_request(const mid& mid, uint64_t count)
    {
        auto params = yhttp::url_encode(
            { { "uid", uid_ }, { "mid", mid }, { "count", count }, { "_", ctx_->uniq_id() } });
        return yhttp::request::GET("/next_message_chunk" + params, ticket_header_);
    }

    context_ptr ctx_;
    uid uid_;
    std::string owner_host_;
    std::string ticket_header_;
    settings_ptr settings_;
};

class pop3_http_mailbox : public http_mailbox
{
public:
    using http_mailbox::http_mailbox;

private:
    yhttp::request make_folders_request() override
    {
        auto params = yhttp::url_encode({ { "uid", uid_ }, { "_", ctx_->uniq_id() } });
        return yhttp::request::GET("/pop3_folders" + params, ticket_header_);
    }

    yhttp::request make_next_message_chunk_request(const mid& mid, uint64_t count) override
    {
        auto params = yhttp::url_encode(
            { { "uid", uid_ }, { "mid", mid }, { "count", count }, { "_", ctx_->uniq_id() } });
        return yhttp::request::GET("/pop3_next_message_chunk" + params, ticket_header_);
    }
};

template <typename Callback>
void call_internal_api_op<Callback>::operator()(yield_context yield_ctx)
{
    reenter(yield_ctx)
    {
        req.url = mailbox->get_owner_address() + orig_url;
        yield client->async_run(ctx, req, yield_ctx.capture(ec, resp));
        if (ec) yield break;

        if (resp.status == 410)
        {
            yield mailbox->refresh_owner(yield_ctx.capture(ec));
            if (ec) yield break;

            req.url = mailbox->get_owner_address() + orig_url;
            yield client->async_run(ctx, req, yield_ctx.capture(ec, resp));
            if (ec) yield break;
        }
    }
    if (yield_ctx.is_complete()) complete();
}

template <typename Callback>
void call_internal_api_op<Callback>::operator()(typename yield_context::exception_type exception)
{
    ec = make_error(exception);
    TASK_LOG(ctx, error) << "call_internal_api_op exception: " << error_message(ec);
    complete();
}

template <typename Callback>
void call_internal_api_op<Callback>::complete()
{
    cb(ec, resp);
}

}

#include <yplatform/unyield.h>
