#pragma once

#include "conversions.h"
#include <common/errors.h>
#include <common/types.h>
#include <ymod_macs/module.h>
#include <yplatform/coroutine.h>
#include <yplatform/yield.h>

namespace collectors::mailbox {

template <typename GetNextEnvelopeChunkFunc>
struct load_next_messages_chunk_op
{
    using yield_ctx = yplatform::yield_context<load_next_messages_chunk_op>;

    ymod_macs::macs_service_ptr macs_service;
    context_ptr context;
    GetNextEnvelopeChunkFunc get_next_envelope_chunk;
    messages_cb handler;

    mail_errors::error_code macs_error = {};
    macs::LabelSet macs_labels = {};
    macs::EnvelopeChunk envelopes = {};
    error ec = {};
    messages res = {};

    void operator()(yield_ctx ctx)
    {
        reenter(ctx)
        {
            yield macs_service->labels().getAllLabels(ctx.capture(macs_error, macs_labels));
            if (macs_error) yield break;

            yield get_next_envelope_chunk(macs_labels, ctx.capture(macs_error, envelopes));
            if (macs_error) yield break;

            auto now = std::time(nullptr);
            for (auto&& envelope : envelopes)
            {
                if (envelope.receiveDate() <= now)
                {
                    res.emplace_back(to_message(envelope, macs_labels));
                }
            }
        }
        if (ctx.is_complete())
        {
            if (macs_error)
            {
                YLOG_CTX_LOCAL(context, error)
                    << "get_next_message_chunk error: " << macs_error.message();
                ec = error(code::macs_error, macs_error.full_message());
            }
            complete();
        }
    }

    void operator()(typename yield_ctx::exception_type exception)
    {
        ec = make_error(exception);
        YLOG_CTX_LOCAL(context, error)
            << "exception during load_next_messages_chunk_op: " << error_message(ec);
        complete();
    }

    void complete()
    {
        handler(ec, res);
    }
};

template <typename GetNextEnvelopeChunkFunc, typename Callback>
void spawn_load_next_messages_chunk_op(
    ymod_macs::macs_service_ptr macs_service,
    context_ptr ctx,
    GetNextEnvelopeChunkFunc&& get_next_envelope_chunk_func,
    Callback&& cb)
{
    using operation_type = load_next_messages_chunk_op<GetNextEnvelopeChunkFunc>;
    auto op = std::make_shared<operation_type>(
        operation_type{ macs_service,
                        ctx,
                        std::forward<GetNextEnvelopeChunkFunc>(get_next_envelope_chunk_func),
                        std::forward<Callback>(cb) });
    yplatform::spawn(op);
}

}
