#include "mailbox.h"
#include "conversions.h"
#include "load_next_messages_chunk_op.h"
#include "store_message_op.h"

#include <ymod_macs/module.h>
// XXX <-- extra free space?
#include <yplatform/find.h>

#include <memory>

namespace collectors::mailbox {

class macs_mailbox
    : public std::enable_shared_from_this<mailbox>
    , public mailbox
{ // XXX order
public:
    macs_mailbox(context_ptr ctx, const uid& uid) : ctx_(ctx), uid_(std::move(uid))
    {
        auto macs = yplatform::find<ymod_macs::module>("macs");
        macs_service_ = macs->get_service(ctx_, uid_);
    }

    void get_folders(const folders_cb& cb) override
    {
        if (ctx_->is_cancelled())
            return cb(code::cancelled_context, {}); // XXX del, check in high-level funcs

        macs_service_->folders().getAllFolders(
            [cb, this, self = shared_from_this()](auto&& ec, auto&& macs_folders) {
                if (ec)
                {
                    YLOG_CTX_LOCAL(ctx_, error) << "load_collectors error: " << ec.message();
                    return cb(code::macs_error, {});
                }

                folders res; // XXX reserve?
                for (auto& [fid, macs_folder] : macs_folders)
                {
                    res.push_back(to_folder(macs_folder));
                }
                cb(code::ok, std::move(res));
            });
    }

    void get_folder_by_type(const std::string& title, const folder_cb& cb) override
    {
        if (ctx_->is_cancelled()) return cb(code::cancelled_context, {});

        macs_service_->folders().getAllFolders(
            [cb, title, this, self = shared_from_this()](auto&& ec, auto&& macs_folders) {
                if (ec)
                {
                    YLOG_CTX_LOCAL(ctx_, error) << "get_folder_by_type error: " << ec.message();
                    return cb(code::macs_error, {});
                }

                auto macs_folder = macs_folders.at(macs::Folder::Symbol::getByTitle(title));
                cb(code::ok, to_folder(macs_folder));
            });
    }

    void get_labels(const labels_cb& cb) override
    {
        if (ctx_->is_cancelled()) return cb(code::cancelled_context, {});

        macs_service_->labels().getAllLabels(
            [cb, this, self = shared_from_this()](auto&& ec, auto&& macs_labels) {
                if (ec)
                {
                    YLOG_CTX_LOCAL(ctx_, error) << "get_labels error: " << ec.message();
                    return cb(code::macs_error, {});
                }

                labels res;
                for (auto& [lid, macs_label] : macs_labels)
                {
                    res.push_back(to_label(macs_label));
                }
                cb(code::ok, std::move(res));
            });
    }

    void get_next_message_chunk(const mid& mid, uint64_t count, const messages_cb& cb) override
    {
        if (ctx_->is_cancelled()) return cb(code::cancelled_context, {});
        spawn_load_next_messages_chunk_op(
            macs_service_,
            ctx_,
            [macs = macs_service_, mid, count](auto macs_labels, auto handler) {
                macs->collectors().getNextEnvelopeChunk(
                    mid, count, std::move(macs_labels), std::move(handler));
            },
            cb);
    }

    void create_folder(const folder& folder, const folder_cb& cb) override
    {
        if (ctx_->is_cancelled()) return cb(code::cancelled_context, {});

        auto symbol = macs::Folder::Symbol::getByTitle(folder.symbol);
        macs_service_->folders().createFolder(
            folder.name,
            folder.parent_fid,
            symbol,
            // XXX do need "this"?
            [cb, this, self = shared_from_this()](auto&& ec, auto&& macs_folder) {
                if (ec)
                {
                    YLOG_CTX_LOCAL(ctx_, error) << "create_folder error: " << ec.message();
                    return cb(code::macs_error, {});
                }

                cb(code::ok, to_folder(macs_folder));
            });
    }

    void create_label(const label& label, const label_cb& cb) override
    {
        if (ctx_->is_cancelled()) return cb(code::cancelled_context, {});

        auto handler = [cb, this, self = shared_from_this()](auto&& ec, auto&& macs_label) {
            if (ec)
            {
                YLOG_CTX_LOCAL(ctx_, error) << "create_label error: " << ec.message();
                return cb(code::macs_error, {});
            }

            cb(code::ok, to_label(macs_label));
        };
        if (label.symbol.empty())
        {
            macs_service_->labels().createLabelWithType(
                label.name, label.color, label.type, handler);
        }
        else
        {
            auto symbol = macs::Label::Symbol::getByTitle(label.symbol);
            macs_service_->labels().createLabel(symbol, handler);
        }
    }

    void store_message(
        const message& msg,
        const std::string& email,
        bool disable_push,
        bool skip_loop_prevention,
        const std::vector<std::string>& rpop_ids,
        const std::string& rpop_info,
        const mid_cb& cb) override
    {
        if (ctx_->is_cancelled()) return cb(code::cancelled_context, "");

        // XXX inconsistent formatting
        auto op = std::make_shared<store_message_op>(
            ctx_, uid_, msg, email, disable_push, skip_loop_prevention, rpop_ids, rpop_info, cb);
        yplatform::spawn(op);
    }

protected:
    context_ptr ctx_;
    uid uid_;
    ymod_macs::macs_service_ptr macs_service_;
};

class pop3_macs_mailbox : public macs_mailbox
{
public:
    using macs_mailbox::macs_mailbox;

    void get_folders(const folders_cb& cb) override
    {
        if (ctx_->is_cancelled()) return cb(code::cancelled_context, {});

        macs_service_->folders().getAllFolders(
            [cb, this, self = shared_from_this()](auto&& ec, auto&& macs_folders) {
                if (ec)
                {
                    YLOG_CTX_LOCAL(ctx_, error) << "get_folders error: " << ec.message();
                    return cb(code::macs_error, {});
                }

                folders res;
                res.reserve(macs_folders.size());
                for (auto& [fid, macs_folder] : macs_folders)
                {
                    res.push_back(to_folder(macs_folder));
                    res.back().skip_messages = !macs_folder.pop3On();
                }
                cb(code::ok, std::move(res));
            });
    }

    void get_next_message_chunk(const mid& mid, uint64_t count, const messages_cb& cb) override
    {
        if (ctx_->is_cancelled()) return cb(code::cancelled_context, {});
        spawn_load_next_messages_chunk_op(
            macs_service_,
            ctx_,
            [macs = macs_service_, mid, count](auto macs_labels, auto handler) {
                macs->collectors().getNextPop3EnvelopeChunk(
                    mid, count, std::move(macs_labels), std::move(handler));
            },
            cb);
    }
};

}
