#include "macs_mailbox.h"
#include "http_mailbox.h"

#include <ymod_mdb_sharder/errors.h>
#include <ymod_mdb_sharder/users_distributor.h>

namespace collectors::mailbox {

mailbox_ptr make_mailbox(context_ptr ctx, const uid& uid)
{
    return std::make_shared<macs_mailbox>(ctx, uid);
}

template <typename Impl>
void make_http_mailbox(
    context_ptr ctx,
    const uid& uid,
    const std::string& host,
    settings_ptr settings,
    const ro_mailbox_cb& cb)
{
    YLOG_CTX_GLOBAL(ctx, info) << "create read_only_mailbox: remote, host=" << host;
    auto tvm = yplatform::find<ymod_tvm::tvm2_module>("tvm_internal");
    std::string service_ticket;
    auto err = tvm->get_service_ticket(settings->internal_api_service_name, service_ticket);
    cb(err, err ? nullptr : std::make_shared<Impl>(ctx, uid, host, service_ticket, settings));
}

void make_read_only_mailbox(
    context_ptr ctx,
    const uid& uid,
    settings_ptr settings,
    const ro_mailbox_cb& cb)
{
    auto distributor = yplatform::find<ymod_mdb_sharder::users_distributor>("users_distributor");
    distributor->get_owner(
        ctx,
        std::stoull(uid),
        [ctx, settings, uid, cb, my_node_id = distributor->my_node_id()](
            error ec, const ymod_mdb_sharder::node_info& node) {
            if (ec == ymod_mdb_sharder::error::user_not_found)
            {
                return cb(code::user_not_found, nullptr);
            }
            else if (ec)
            {
                return cb(ec, nullptr);
            }

            if (node.id == my_node_id)
            {
                YLOG_CTX_GLOBAL(ctx, info) << "create read_only_mailbox: local";
                cb(code::ok, std::make_shared<macs_mailbox>(ctx, uid));
            }
            else
            {
                make_http_mailbox<http_mailbox>(ctx, uid, node.host, settings, cb);
            }
        });
}

void make_read_only_pop3_mailbox(
    context_ptr ctx,
    const uid& uid,
    settings_ptr settings,
    const ro_mailbox_cb& cb)
{
    auto distributor = yplatform::find<ymod_mdb_sharder::users_distributor>("users_distributor");
    distributor->get_owner(
        ctx,
        std::stoull(uid),
        [ctx, settings, uid, cb, my_node_id = distributor->my_node_id()](auto&& ec, auto&& node) {
            if (ec == ymod_mdb_sharder::error::user_not_found)
            {
                return cb(code::user_not_found, nullptr);
            }
            else if (ec)
            {
                return cb(ec, nullptr);
            }

            if (node.id == my_node_id)
            {
                YLOG_CTX_GLOBAL(ctx, info) << "create read_only_mailbox: local";
                cb(code::ok, std::make_shared<pop3_macs_mailbox>(ctx, uid));
            }
            else
            {
                make_http_mailbox<pop3_http_mailbox>(ctx, uid, node.host, settings, cb);
            }
        });
}

void make_local_read_only_mailbox(
    context_ptr ctx,
    const uid& uid,
    settings_ptr settings,
    const ro_mailbox_cb& cb)
{
    auto distributor = yplatform::find<ymod_mdb_sharder::users_distributor>("users_distributor");
    auto my_node_id = distributor->my_node_id();
    distributor->get_owner(
        ctx,
        std::stoull(uid),
        [ctx, settings, uid, cb, my_node_id](error ec, const ymod_mdb_sharder::node_info& node) {
            if (ec) return cb(ec, nullptr);

            if (node.id != my_node_id)
            {
                cb(code::user_released, {});
            }
            else
            {
                YLOG_CTX_GLOBAL(ctx, info) << "create read_only_mailbox: local";
                cb(ec, std::make_shared<macs_mailbox>(ctx, uid));
            }
        });
}

void make_local_read_only_pop3_mailbox(
    context_ptr ctx,
    const uid& uid,
    settings_ptr settings,
    const ro_mailbox_cb& cb)
{
    auto distributor = yplatform::find<ymod_mdb_sharder::users_distributor>("users_distributor");
    auto my_node_id = distributor->my_node_id();
    distributor->get_owner(
        ctx, std::stoull(uid), [ctx, settings, uid, cb, my_node_id](auto&& ec, auto&& node) {
            if (ec) return cb(ec, nullptr);

            if (node.id != my_node_id)
            {
                cb(code::user_released, {});
            }
            else
            {
                YLOG_CTX_GLOBAL(ctx, info) << "create read_only_mailbox: local";
                cb(ec, std::make_shared<pop3_macs_mailbox>(ctx, uid));
            }
        });
}

}
