#pragma once

#include <common/json.h>

#include <ymod_httpclient/cluster_client.h>
#include <ymod_httpclient/url_encode.h>

#include <yplatform/coroutine.h>
#include <yplatform/json.h>
#include <yplatform/yield.h>

namespace collectors::mailbox {

using yplatform::json_value;
using yplatform::json_type;

struct store_message_op
{
    using yield_ctx = yplatform::yield_context<store_message_op>;

    store_message_op(
        context_ptr context,
        const uid& uid,
        const message& msg,
        const std::string& email,
        bool disable_push,
        bool skip_loop_prevention,
        const std::vector<std::string>& rpop_ids,
        const std::string& rpop_info,
        const mid_cb& handler)
        : context(context)
        , uid(uid)
        , msg(msg)
        , email(email)
        , disable_push(disable_push)
        , skip_loop_prevention(skip_loop_prevention)
        , rpop_ids(rpop_ids)
        , rpop_info(rpop_info)
        , handler(handler)
    {
    }

    void operator()(yield_ctx ctx)
    {
        reenter(ctx)
        {
            yield download_body(msg.stid, ctx.capture(ec, http_resp));
            if (ec || bad_http_status(http_resp.status))
            {
                ec = error(code::download_message_error, get_download_error_reason(ec, http_resp));
                yield break;
            }

            body = http_resp.body;
            add_headers(body);

            yield store_message(ctx.capture(ec, http_resp));
            if (ec || bad_http_status(http_resp.status))
            {
                ec = error(
                    (http_resp.status == 409 ? code::message_is_duplicate :
                                               code::store_message_error),
                    get_store_error_reason(ec, http_resp));
                yield break;
            }

            ec = process_store_response(http_resp, mid);
            if (ec)
            {
                yield break;
            }
        }
        if (ctx.is_complete()) complete();
    }

    void operator()(yield_ctx::exception_type exception)
    {
        ec = make_error(exception);
        YLOG_CTX_LOCAL(context, error)
            << "exception during store_message_op: " << error_message(ec);
        complete();
    }

    void complete()
    {
        handler(ec, mid);
    }

    template <typename Handler>
    void download_body(const std::string& stid, const Handler& handler)
    {
        auto client = yplatform::find<yhttp::cluster_client>("storage_client");
        auto req = yhttp::request::GET("/gate/get/" + stid);
        client->async_run(context, req, handler);
    }

    bool bad_http_status(int status)
    {
        return status / 100 != 2;
    }

    std::string get_download_error_reason(error ec, const yhttp::response& http_resp)
    {
        if (ec) return ec.message();
        return http_resp.body;
    }

    std::string get_store_error_reason(error ec, const yhttp::response& http_resp)
    {
        if (ec) return ec.message();

        json_value response;
        if (auto error = response.parse(http_resp.body, json_type::tobject))
        {
            TASK_LOG(context, error) << "store_message_op json parse error: " << *error;
            return http_resp.body;
        }

        return response.has_member("message") ? response["message"].to_string() : http_resp.body;
    }

    void add_headers(std::string& body)
    {
        auto crlf = "\r\n";
        std::string headers;
        for (auto&& id : rpop_ids)
        {
            headers += "X-yandex-rpop-id: " + id + crlf;
        }
        headers += "X-yandex-rpop-info: " + rpop_info + crlf;
        body = headers + body;
    }

    template <typename Handler>
    void store_message(const Handler& handler)
    {
        auto client = yplatform::find<yhttp::cluster_client>("nw_client");
        auto params =
            yhttp::url_encode({ { "service", "collectors" }, { "uid", uid }, { "fid", msg.fid } });
        auto headers = "X-Request-Id: " + context->uniq_id() + "\r\n";
        auto multipart = yhttp::body_mixed(
            { yhttp::json_part(make_json_args()), yhttp::rfc822_part(std::move(body)) });
        auto req =
            yhttp::request::MPOST("/mail/store" + params, std::move(headers), std::move(multipart));
        client->async_run(context, std::move(req), handler);
    }

    std::string make_json_args()
    {
        json::value res;
        res["options"] = make_options_arg();

        json::value mail_info;
        mail_info["received_date"] = msg.date;
        mail_info["labels"] = make_labels_arg();
        res["mail_info"] = mail_info;

        json::value user_info;
        user_info["email"] = email;
        res["user_info"] = user_info;

        return json::to_string(res);
    }

    json::value make_options_arg()
    {
        json::value options;
        options["detect_spam"] = false;
        options["detect_virus"] = false;
        options["detect_loop"] = !skip_loop_prevention;
        options["enable_push"] = !disable_push;
        options["use_filters"] = true;
        options["allow_duplicates"] = false;
        return options;
    }

    json::value make_labels_arg()
    {
        json::value symbols(Json::arrayValue);
        json::value lids(Json::arrayValue);
        for (auto& label : msg.labels)
        {
            if (label.symbol.empty())
            {
                lids.append(label.lid);
            }
            else
            {
                symbols.append(label.symbol);
            }
        }

        json::value labels;
        labels["symbol"] = symbols;
        labels["lids"] = lids;
        labels["imap"] = Json::arrayValue;
        labels["system"] = Json::arrayValue;
        return labels;
    }

    error process_store_response(const yhttp::response& http_resp, std::string& mid)
    {
        json_value response;
        if (auto error = response.parse(http_resp.body, json_type::tobject))
        {
            TASK_LOG(context, error) << "store_message_op json parse error: " << *error;
            return code::store_message_error;
        }

        mid = response["mid"].to_string();
        return {};
    }

    context_ptr context;
    uid uid;
    message msg;
    std::string email;
    bool disable_push;
    bool skip_loop_prevention;
    std::vector<std::string> rpop_ids;
    std::string rpop_info;
    std::string body;
    std::string mid;
    mid_cb handler;

    error ec;
    yhttp::response http_resp;
};

}

#include <yplatform/unyield.h>
