#pragma once

#include "client.h"

#include <common/server_info.h>

#include <ymod_blackbox/client.h>
#include <ymod_httpclient/cluster_client.h>
#include <yplatform/find.h>

#include <butil/network/idn.h>

namespace collectors::passport {

const std::string SID = "2";

class client_impl
    : public std::enable_shared_from_this<client_impl>
    , public client
{
    using info_response = ymod_blackbox::info_response;
    using session_id_response = ymod_blackbox::session_id_response;
    using uid_option = ymod_blackbox::uid_option;
    using email_list_option = ymod_blackbox::email_list_option;
    using option = ymod_blackbox::option;

public:
    client_impl(context_ptr ctx) : ctx(ctx)
    {
        bbclient = yplatform::find<ymod_blackbox::client>("bbclient");
        validator_client = yplatform::find<yhttp::cluster_client>("validator_client");
    }

    virtual ~client_impl()
    {
    }

    void get_userinfo_by_suid(const suid& suid, const user_info_cb& cb) override
    {
        bbclient->async_info_suid(
            suid,
            SID,
            my_ip,
            [cb, this, self = shared_from_this()](
                ymod_blackbox::error error, const info_response& resp) {
                if (error)
                {
                    YLOG_CTX_LOCAL(ctx, error)
                        << "get_userinfo_by_suid error: " << error.code.message()
                        << ", ext_reason: " << error.ext_reason;
                    return cb(code::passport_error, {});
                }
                auto info = parse_user_info(resp);
                cb(verify_parsed_info(info), std::move(info));
            },
            { option::get_yandex_emails });
    }

    void get_userinfo_by_uid(const uid& uid, const user_info_cb& cb) override
    {
        bbclient->async_info(
            uid,
            my_ip,
            [cb, this, self = shared_from_this()](
                ymod_blackbox::error error, const info_response& resp) {
                if (error)
                {
                    YLOG_CTX_LOCAL(ctx, error)
                        << "get_userinfo_by_uid error: " << error.code.message()
                        << ", ext_reason: " << error.ext_reason;
                    return cb(code::passport_error, {});
                }
                auto info = parse_user_info(resp);
                cb(verify_parsed_info(info), std::move(info));
            },
            { option::get_yandex_emails });
    }

    void get_userinfo_by_login(const std::string& login, const user_info_cb& cb) override
    {
        bbclient->async_info_login(
            login,
            "",
            my_ip,
            [cb, this, self = shared_from_this()](
                ymod_blackbox::error error, const info_response& resp) {
                if (error)
                {
                    YLOG_CTX_LOCAL(ctx, error)
                        << "get_userinfo_by_login error: " << error.code.message()
                        << ", ext_reason: " << error.ext_reason;
                    return cb(code::passport_error, {});
                }
                auto info = parse_user_info(resp);
                cb(verify_parsed_info(info), std::move(info));
            },
            { option::get_yandex_emails });
    }

    void check_auth_token(const std::string& token, address user_addr, const user_info_cb& cb)
        override
    {
        if (user_addr.ip.empty())
        {
            user_addr.ip = my_ip;
        }
        if (!user_addr.port)
        {
            user_addr.port = my_port;
        }
        bbclient->async_oauth(
            token,
            user_addr,
            [cb, this, self = shared_from_this()](
                ymod_blackbox::error error, const session_id_response& resp) {
                if (error)
                {
                    YLOG_CTX_LOCAL(ctx, error) << "check_auth_token error: " << error.code.message()
                                               << ", ext_reason: " << error.ext_reason;
                    return cb(code::passport_error, {});
                }

                if (resp.status != session_id_response::valid)
                {
                    YLOG_CTX_LOCAL(ctx, error) << "check_auth_token bad status: " << resp.status
                                               << ", message: " << resp.message();
                    return cb(code::invalid_auth_token, {});
                }

                auto info = parse_user_info(resp);
                cb(verify_parsed_info(info), std::move(info));
            },
            { option::get_yandex_emails });
    }

    void add_alias(
        const uid& uid,
        const std::string& alias,
        const std::string& consumer,
        const no_data_cb& cb) override
    {
        auto params = yhttp::url_encode({ { "op", "addrpop" },
                                          { "from", consumer },
                                          { "uid", uid },
                                          { "email", encode_alias(alias) } });

        auto req = yhttp::request::GET("/validate.xml" + params);
        validator_client->async_run(
            ctx, req, [cb, this, self = shared_from_this()](error ec, const yhttp::response& resp) {
                if (ec)
                {
                    TASK_LOG(ctx, error) << "add_alias error: " << ec.message();
                    return cb(code::validator_error);
                }

                if (resp.status / 100 != 2)
                {
                    YLOG_CTX_LOCAL(ctx, error)
                        << "add_alias bad status: " << resp.status << ", body" << resp.body;
                    return cb(code::validator_error);
                }

                auto tag_pos = resp.body.find("validator-rpop-added");
                if (tag_pos != std::string::npos)
                {
                    cb(code::ok);
                }
                else
                {
                    YLOG_CTX_LOCAL(ctx, error) << "add_alias error: " << resp.body;
                    cb(code::validator_error);
                }
            });
    }

    void remove_alias(
        const uid& uid,
        const std::string& alias,
        const std::string& consumer,
        const no_data_cb& cb) override
    {
        auto params = yhttp::url_encode({ { "op", "deleterpop" },
                                          { "from", consumer },
                                          { "uid", uid },
                                          { "email", encode_alias(alias) } });

        auto req = yhttp::request::GET("/validate.xml" + params);
        validator_client->async_run(
            ctx, req, [cb, this, self = shared_from_this()](error ec, const yhttp::response& resp) {
                if (ec)
                {
                    TASK_LOG(ctx, error) << "remove_alias error: " << ec.message();
                    return cb(code::validator_error);
                }

                if (resp.status / 100 != 2)
                {
                    YLOG_CTX_LOCAL(ctx, error)
                        << "remove_alias bad status: " << resp.status << ", body" << resp.body;
                    return cb(code::validator_error);
                }

                auto tag_pos = resp.body.find("validator-address-deleted");
                if (tag_pos != std::string::npos)
                {
                    cb(code::ok);
                }
                else
                {
                    YLOG_CTX_LOCAL(ctx, error) << "remove_alias error: " << resp.body;
                    cb(code::validator_error);
                }
            });
    }

    void get_suid(const uid& uid, const suid_cb& cb) override
    {
        bbclient->async_info(
            uid,
            my_ip,
            [cb, this, self = shared_from_this()](
                ymod_blackbox::error error, const info_response& resp) {
                if (error)
                {
                    YLOG_CTX_LOCAL(ctx, error) << "get_suid error: " << error.code.message()
                                               << ", ext_reason: " << error.ext_reason;
                    return cb(code::passport_error, {});
                }

                cb(code::ok, resp["subscription.suid.2"]);
            },
            {},
            { ymod_blackbox::db_field("subscription.suid.2") });
    }

private:
    std::string encode_alias(const std::string& alias)
    {
        try
        {
            return idna::encode_addrspec(alias);
        }
        catch (const std::exception& /*e*/)
        {
            return alias;
        }
    }

    template <typename bb_response>
    user_info parse_user_info(const bb_response& resp)
    {
        user_info res;

        uid_option uid_opt;
        resp >> uid_opt;
        res.uid = uid_opt.uid;

        email_list_option emails_opt;
        resp >> emails_opt;
        for (auto&& email : emails_opt.items)
        {
            if (res.email.empty() && email.native && email.validated)
            {
                // Use it as fallback, if default will not be found
                res.email = email.address;
            }
            if (email.def && email.native && email.validated)
            {
                res.email = email.address;
                break;
            }
        }
        return res;
    }

    error verify_parsed_info(const user_info& info)
    {
        auto err = code::ok;
        if (info.uid.empty())
        {
            err = code::user_not_found;
        }
        else if (info.email.empty())
        {
            err = code::empty_email;
        }
        return err;
    }

    context_ptr ctx;
    boost::shared_ptr<ymod_blackbox::client> bbclient;
    boost::shared_ptr<yhttp::cluster_client> validator_client;
};

}
