#include "repository_impl.h"

#include <macs_pg/collectors/collector.h>

#include <ymod_macs/module.h>
#include <yplatform/find.h>

namespace collectors::streamer::meta {

collector_state state_from_macs_enum(macs::CollectorState macs_state)
{
    switch (macs_state)
    {
    case macs::CollectorState::undefined:
        return collector_state::created;
    case macs::CollectorState::created:
        return collector_state::created;
    case macs::CollectorState::ready:
        return collector_state::ready;
    case macs::CollectorState::disabled:
        return collector_state::disabled;
    case macs::CollectorState::deleted:
        return collector_state::deleted;
    case macs::CollectorState::migrated:
        return collector_state::migrated;
    case macs::CollectorState::unmigrated:
        return collector_state::unmigrated;
    default:
        return collector_state::created;
    }
}

macs::CollectorState state_to_macs_enum(collector_state state)
{
    switch (state)
    {
    case collector_state::created:
        return macs::CollectorState::created;
    case collector_state::ready:
        return macs::CollectorState::ready;
    case collector_state::disabled:
        return macs::CollectorState::disabled;
    case collector_state::deleted:
        return macs::CollectorState::deleted;
    case collector_state::migrated:
        return macs::CollectorState::migrated;
    case collector_state::unmigrated:
        return macs::CollectorState::unmigrated;
    default:
        return macs::CollectorState::undefined;
    }
}

collector_info to_collector_info(const macs::Collector& macs_collector)
{
    collector_info res(macs_collector.collectorId(), macs_collector.uid(), macs_collector.srcUid());
    res.auth_token = macs_collector.authToken();
    res.creation_ts = macs_collector.creationTs();
    res.root_folder_id = normalize_id(macs_collector.rootFolderId());
    res.label_id = normalize_id(macs_collector.labelId());
    res.ignore_folders_struct = macs_collector.ignoreFoldersStruct();
    res.last_mid = normalize_id(macs_collector.lastMid());
    res.skipped_mids = macs_collector.skippedMids();
    res.state = state_from_macs_enum(macs_collector.state());
    res.old_popid = macs_collector.oldPopid();
    res.original_server = macs_collector.originalServer().empty() ? "imap.yandex.ru" :
                                                                    macs_collector.originalServer();
    res.migration_target_state = state_from_macs_enum(macs_collector.migrationTargetState());
    return res;
}

void load_collectors(
    const std::string& shard_id,
    const uids& uids,
    const collector_info_chunk_cb& cb)
{
    auto macs = yplatform::find<ymod_macs::module>("macs");
    auto shard = macs->get_shard(shard_id);
    shard->collectors().getCollectors(uids, [cb](auto&& ec, auto&& macs_collectors) {
        if (ec)
        {
            YLOG_G(error) << "load_collectors error: " << ec.message();
            return cb(code::macs_error, {});
        }

        collector_info_chunk chunk;
        std::transform(
            macs_collectors.begin(),
            macs_collectors.end(),
            std::back_inserter(chunk),
            to_collector_info);
        cb(code::ok, chunk);
    });
}

void create_collector(
    context_ptr ctx,
    const new_collector_draft& draft,
    const collector_info_cb& cb)
{
    macs::CollectorFactory factory;
    factory.srcUid(draft.src_uid)
        .authToken(draft.auth_token)
        .rootFolderId(draft.root_folder_id)
        .labelId(draft.label_id);
    auto macs = yplatform::find<ymod_macs::module>("macs");
    auto service = macs->get_service(ctx, draft.dst_uid);
    service->collectors().createCollector(std::move(factory), [cb, ctx](auto&& ec, auto&& res) {
        if (ec)
        {
            auto message = ec.message();
            LERR_(ctx) << "create_collector error: " << message;
            if (boost::algorithm::contains(message, "Can't create collector from myself"))
            {
                return cb(code::collector_from_himself, {});
            }
            if (boost::algorithm::contains(message, "Can't create collector - already exists"))
            {
                return cb(code::duplicate_collector, {});
            }
            return cb(code::macs_error, {});
        }
        cb(code::ok, to_collector_info(res.collector));
    });
}

void migrate_collector(
    context_ptr ctx,
    const migrated_collector_draft& draft,
    const collector_info_cb& cb)
{
    macs::CollectorFactory factory;
    factory.srcUid(draft.src_uid)
        .authToken(draft.auth_token)
        .ignoreFoldersStruct(draft.ignore_folders_struct)
        .lastMid(draft.last_mid)
        .skippedMids(draft.skipped_mids)
        .oldPopid(macs::OldPopid(draft.old_popid))
        .creationTs(draft.creation_ts)
        .originalServer(draft.original_server)
        .migrationTargetState(state_to_macs_enum(draft.target_state));
    auto macs = yplatform::find<ymod_macs::module>("macs");
    auto service = macs->get_service(ctx, draft.dst_uid);
    service->collectors().migrateCollector(std::move(factory), [cb, ctx](auto&& ec, auto&& res) {
        if (ec)
        {
            auto message = ec.message();
            LERR_(ctx) << "migrate_collector error: " << message;
            if (boost::algorithm::contains(message, "Can't create collector from myself"))
            {
                return cb(code::collector_from_himself, {});
            }
            if (boost::algorithm::contains(message, "Can't create collector - already exists"))
            {
                return cb(code::duplicate_collector, {});
            }
            return cb(code::macs_error, {});
        }
        cb(code::ok, to_collector_info(res.collector));
    });
}

repository_ptr make_repository(
    context_ptr ctx,
    collector_info_ptr data,
    boost::asio::io_context* io)
{
    return std::make_shared<repository_impl>(ctx, data, io);
}

collector_logging_info get_logging_info(repository_ptr repo)
{
    return collector_logging_info{ .dst_uid = repo->dst_uid(),
                                   .src_uid = repo->src_uid(),
                                   .id = repo->global_id() };
}

collector_logging_info get_logging_info(const collector_info& collector_info)
{
    return collector_logging_info{ .dst_uid = collector_info.dst_uid,
                                   .src_uid = collector_info.src_uid,
                                   .id = { collector_info.dst_uid, collector_info.id } };
}

}
