#pragma once

#include <common/types.h>
#include <common/collector_info.h>

#include <memory>

namespace collectors::streamer::meta {

class repository
{
public:
    repository(collector_info_ptr collector_info, boost::asio::io_context* io)
        : collector_info(collector_info), io(io)
    {
    }

    const collector_id& collector_id()
    {
        return collector_info->id;
    }
    const uid& dst_uid()
    {
        return collector_info->dst_uid;
    }
    const uid& src_uid()
    {
        return collector_info->src_uid;
    }
    const std::string& auth_token()
    {
        ensure_thread_safety();
        return collector_info->auth_token;
    }
    const std::time_t& creation_ts()
    {
        ensure_thread_safety();
        return collector_info->creation_ts;
    }
    const fid& root_folder_id()
    {
        ensure_thread_safety();
        return collector_info->root_folder_id;
    }
    const lid& label_id()
    {
        ensure_thread_safety();
        return collector_info->label_id;
    }
    bool ignore_folders_struct()
    {
        ensure_thread_safety();
        return collector_info->ignore_folders_struct;
    }
    const mid& last_mid()
    {
        ensure_thread_safety();
        return collector_info->last_mid;
    }
    const mids& skipped_mids()
    {
        ensure_thread_safety();
        return collector_info->skipped_mids;
    }
    uint64_t old_popid()
    {
        ensure_thread_safety();
        return collector_info->old_popid;
    }
    std::string original_server()
    {
        ensure_thread_safety();
        return collector_info->original_server;
    }
    collector_state state()
    {
        ensure_thread_safety();
        return collector_info->state;
    }
    collector_state migration_target_state()
    {
        ensure_thread_safety();
        return collector_info->migration_target_state;
    }

    std::time_t last_run_ts()
    {
        ensure_thread_safety();
        return collector_info->last_run_ts;
    }

    global_collector_id global_id()
    {
        return { collector_info->dst_uid, collector_info->id };
    }

    virtual void update_last_mid(const mid& last_mid, const no_data_cb& cb) = 0;
    virtual void update_skipped_mids(const mids& skipped_mids, const no_data_cb& cb) = 0;
    virtual void reset_token(const no_data_cb& cb) = 0;
    virtual void update_state(collector_state state, const no_data_cb& cb) = 0;
    virtual void update_migration_target_state(collector_state state, const no_data_cb& cb) = 0;
    virtual void edit(
        const std::optional<std::string>& auth_token,
        const std::optional<fid>& root_folder_id,
        const std::optional<lid>& label_id,
        const no_data_cb& cb) = 0;
    virtual void reset_collector(
        const std::string& auth_token,
        const fid& root_folder_id,
        const lid& label_id,
        const no_data_cb& cb) = 0;
    virtual void delete_collector(const no_data_cb& cb) = 0;
    virtual void update_last_run_ts(std::time_t ts) = 0;

protected:
    void ensure_thread_safety()
    {
        if (!io->get_executor().running_in_this_thread())
        {
            throw std::runtime_error("not thread-safe get operation in meta::repository");
        }
    }

    collector_info_ptr collector_info;
    boost::asio::io_context* io;
};

using repository_ptr = std::shared_ptr<repository>;

void load_collectors(
    const std::string& shard_id,
    const uids& uids,
    const collector_info_chunk_cb& cb);
void create_collector(
    context_ptr ctx,
    const new_collector_draft& draft,
    const collector_info_cb& cb);
void migrate_collector(
    context_ptr ctx,
    const migrated_collector_draft& draft,
    const collector_info_cb& cb);

repository_ptr make_repository(
    context_ptr ctx,
    collector_info_ptr collector_info,
    boost::asio::io_context* io);

collector_logging_info get_logging_info(repository_ptr repo);
collector_logging_info get_logging_info(const collector_info& collector_info);

}
