#pragma once

#include "repository.h"

#include <ymod_macs/module.h>
#include <yplatform/find.h>

namespace collectors::streamer::meta {

collector_state state_from_macs_enum(macs::CollectorState macs_state);
macs::CollectorState state_to_macs_enum(collector_state state);

class repository_impl
    : public std::enable_shared_from_this<repository_impl>
    , public repository
{
public:
    repository_impl(context_ptr ctx, collector_info_ptr data, boost::asio::io_context* io)
        : repository(data, io), ctx_(ctx)
    {
        auto macs = yplatform::find<ymod_macs::module>("macs");
        macs_service_ = macs->get_service(ctx_, collector_info->dst_uid);
    }

    virtual ~repository_impl()
    {
    }

    void update_last_mid(const mid& last_mid, const no_data_cb& cb) override
    {
        if (ctx_->is_cancelled()) return cb(code::cancelled_context);

        macs_service_->collectors().updateLastMid(
            macs::CollectorId(collector_id()),
            last_mid,
            io->wrap([last_mid, cb, this, self = shared_from_this()](auto&& ec) {
                if (ec)
                {
                    YLOG_CTX_LOCAL(ctx_, error) << "update_last_mid error: " << ec.message();
                    return cb(code::macs_error);
                }

                collector_info->last_mid = last_mid;
                cb(code::ok);
            }));
    }

    void update_skipped_mids(const mids& skipped_mids, const no_data_cb& cb) override
    {
        if (ctx_->is_cancelled()) return cb(code::cancelled_context);

        macs_service_->collectors().updateSkippedMids(
            macs::CollectorId(collector_id()),
            skipped_mids,
            io->wrap([skipped_mids, cb, this, self = shared_from_this()](auto&& ec) {
                if (ec)
                {
                    YLOG_CTX_LOCAL(ctx_, error) << "update_skipped_mids error: " << ec.message();
                    return cb(code::macs_error);
                }

                collector_info->skipped_mids = skipped_mids;
                cb(code::ok);
            }));
    }

    void reset_token(const no_data_cb& cb) override
    {
        if (ctx_->is_cancelled()) return cb(code::cancelled_context);

        macs_service_->collectors().resetToken(
            macs::CollectorId(collector_id()),
            io->wrap([cb, this, self = shared_from_this()](auto&& ec) {
                if (ec)
                {
                    YLOG_CTX_LOCAL(ctx_, error) << "reset_token error: " << ec.message();
                    return cb(code::macs_error);
                }

                collector_info->auth_token = "";
                cb(code::ok);
            }));
    }

    void update_state(collector_state state, const no_data_cb& cb) override
    {
        if (ctx_->is_cancelled()) return cb(code::cancelled_context);

        macs_service_->collectors().updateState(
            macs::CollectorId(collector_id()),
            state_to_macs_enum(state),
            io->wrap([state, cb, this, self = shared_from_this()](auto&& ec, auto&& /*revision*/) {
                if (ec)
                {
                    YLOG_CTX_LOCAL(ctx_, error) << "update_state error: " << ec.message();
                    return cb(code::macs_error);
                }

                collector_info->state = state;
                cb(code::ok);
            }));
    }

    void update_migration_target_state(collector_state state, const no_data_cb& cb) override
    {
        if (ctx_->is_cancelled()) return cb(code::cancelled_context);

        macs_service_->collectors().updateMigrationTargetState(
            macs::CollectorId(collector_id()),
            state_to_macs_enum(state),
            io->wrap([state, cb, this, self = shared_from_this()](auto&& ec, auto&& /*revision*/) {
                if (ec)
                {
                    YLOG_CTX_LOCAL(ctx_, error)
                        << "update_migration_target_state error: " << ec.message();
                    return cb(code::macs_error);
                }

                collector_info->migration_target_state = state;
                cb(code::ok);
            }));
    }

    void edit(
        const std::optional<std::string>& auth_token,
        const std::optional<fid>& root_folder_id,
        const std::optional<lid>& label_id,
        const no_data_cb& cb) override
    {
        if (ctx_->is_cancelled()) return cb(code::cancelled_context);

        auto factory = std::make_shared<macs::CollectorFactory>();
        (*factory)
            .collectorId(macs::CollectorId(collector_id()))
            .authToken(auth_token ? *auth_token : collector_info->auth_token)
            .rootFolderId(root_folder_id ? *root_folder_id : collector_info->root_folder_id)
            .labelId(label_id ? *label_id : collector_info->label_id);
        auto self = shared_from_this();
        macs_service_->collectors().updateCollector(
            *factory, io->wrap([factory, cb, this, self](auto&& ec, auto&& /*revision*/) {
                if (ec)
                {
                    YLOG_CTX_LOCAL(ctx_, error) << "edit error: " << ec.message();
                    return cb(code::macs_error);
                }

                auto& product = factory->product();
                collector_info->auth_token = product.authToken();
                collector_info->root_folder_id = product.rootFolderId();
                collector_info->label_id = product.labelId();
                cb(code::ok);
            }));
    }

    void reset_collector(
        const std::string& auth_token,
        const fid& root_folder_id,
        const lid& label_id,
        const no_data_cb& cb) override
    {
        if (ctx_->is_cancelled()) return cb(code::cancelled_context);

        auto factory = std::make_shared<macs::CollectorFactory>();
        (*factory)
            .collectorId(macs::CollectorId(collector_id()))
            .state(macs::CollectorState::created)
            .authToken(auth_token)
            .rootFolderId(root_folder_id)
            .labelId(label_id)
            .lastMid(EMPTY_ID)
            .skippedMids(mids())
            .creationTs(std::time(nullptr));

        macs_service_->collectors().resetCollector(
            *factory,
            io->wrap(
                [factory, cb, this, self = shared_from_this()](auto&& ec, auto&& /*revision*/) {
                    if (ec)
                    {
                        YLOG_CTX_LOCAL(ctx_, error) << "reset_collector error: " << ec.message();
                        return cb(code::macs_error);
                    }

                    auto&& product = factory->product();
                    collector_info->state = state_from_macs_enum(product.state());
                    collector_info->auth_token = product.authToken();
                    collector_info->root_folder_id = product.rootFolderId();
                    collector_info->label_id = product.labelId();
                    collector_info->last_mid = product.lastMid();
                    collector_info->skipped_mids = product.skippedMids();
                    collector_info->creation_ts = product.creationTs();
                    cb(code::ok);
                }));
    }

    void delete_collector(const no_data_cb& cb) override
    {
        if (ctx_->is_cancelled()) return cb(code::cancelled_context);

        macs_service_->collectors().deleteCollector(
            macs::CollectorId(collector_id()),
            io->wrap([cb, this, self = shared_from_this()](auto&& ec, auto&& /*revision*/) {
                if (ec)
                {
                    YLOG_CTX_LOCAL(ctx_, error) << "delete_collector error: " << ec.message();
                    return cb(code::macs_error);
                }
                cb(code::ok);
            }));
    }

    void update_last_run_ts(std::time_t ts) override
    {
        collector_info->last_run_ts = ts;
    }

private:
    context_ptr ctx_;
    ymod_macs::macs_service_ptr macs_service_;
};

}
