#pragma once

#include "meta.h"
#include "planner.h"
#include "streamers_loader.h"
#include "streamer_impl.h"

#include <common/errors.h>
#include <common/types.h>

#include <ymod_macs/module.h>
#include <ymod_mdb_sharder/users_distributor.h>
#include <ymod_mdb_sharder/shards_distributor.h>
#include <yplatform/module.h>
#include <yplatform/reactor.h>
#include <yplatform/util/tuple_unpack.h>

#include <memory>

namespace collectors::streamer {

using collector_info_cb = std::function<void(error, const collector_info&)>;

// XXX comment or reference what is streamer
class module : public yplatform::module
{
    using planner_type = planner<global_collector_id>;
    using planner_ptr = std::shared_ptr<planner_type>;
    using loader_type = streamers_loader<decltype(&meta::load_collectors)>;
    using loader_ptr = std::shared_ptr<loader_type>;
    using streamer_type =
        detail::streamer_impl<planner_ptr, loader_ptr, boost::shared_ptr<ymod_macs::module>>;

public:
    template <typename Callback>
    void update_collector_state(
        context_ptr ctx,
        const global_collector_id& id,
        collector_state state,
        const Callback& user_cb)
    {
        impl->post_operation(
            ctx,
            id,
            operation::edit,
            [id, state, user_cb](context_ptr ctx, streamer_data_ptr data, const no_data_cb& op_cb) {
                auto cb = multi_cb{ user_cb, op_cb };
                if (data && data->collector_info.state == collector_state::deleted)
                {
                    data = nullptr;
                }
                if (!data) return cb(code::streamer_data_not_found);

                YLOG_CTX_GLOBAL(ctx, info)
                    << "user operation " << meta::get_logging_info(data->collector_info).to_string()
                    << " op=update_collector_state";
                auto meta = make_meta_repo(ctx, data);
                meta->update_state(state, cb);
            });
    }

    template <typename Callback>
    void edit_collector(
        context_ptr ctx,
        const global_collector_id& id,
        const std::optional<std::string>& auth_token,
        const std::optional<fid>& root_folder_id,
        const std::optional<lid>& label_id,
        const Callback& user_cb)
    {
        auto args = std::tuple(auth_token, root_folder_id, label_id);
        impl->post_operation(
            ctx,
            id,
            operation::edit,
            [id, user_cb, args = std::move(args)](
                context_ptr ctx, streamer_data_ptr data, const no_data_cb& op_cb) {
                auto cb = multi_cb{ user_cb, op_cb };
                if (data && data->collector_info.state == collector_state::deleted)
                {
                    data = nullptr;
                }
                if (!data) return cb(code::streamer_data_not_found);

                YLOG_CTX_GLOBAL(ctx, info)
                    << "user operation " << meta::get_logging_info(data->collector_info).to_string()
                    << "op=edit_collector";
                auto meta = make_meta_repo(ctx, data);
                auto full_args = std::tuple_cat(std::tuple(meta), args);
                yplatform::util::call_with_tuple_args(
                    std::mem_fn(&meta::repository::edit), full_args, cb);
            });
    }

    template <typename Callback>
    void delete_collector(context_ptr ctx, const global_collector_id& id, const Callback& user_cb)
    {
        impl->post_operation(
            ctx,
            id,
            operation::edit,
            [id, user_cb](context_ptr ctx, streamer_data_ptr data, const no_data_cb& op_cb) {
                auto cb = multi_cb{ user_cb, op_cb };
                if (data && data->collector_info.state == collector_state::deleted)
                {
                    data = nullptr;
                }
                if (!data) return cb(code::streamer_data_not_found);

                YLOG_CTX_GLOBAL(ctx, info)
                    << "user operation " << meta::get_logging_info(data->collector_info).to_string()
                    << "op=delete_collector";
                auto meta = make_meta_repo(ctx, data);
                meta->update_state(collector_state::deleted, cb);
            });
    }

    template <typename Callback>
    void create_collector(context_ptr ctx, const new_collector_draft& draft, Callback&& cb)
    {
        auto self = shared_from_this();
        post_operation(
            ctx,
            draft.dst_uid,
            draft.src_uid,
            [draft, user_cb = std::forward<Callback>(cb), this, self](
                context_ptr ctx, streamer_data_ptr data, const no_data_cb& op_cb) {
                auto cb = multi_cb{ user_cb, [op_cb](error ec, collector_id /*collector_id*/) {
                                       op_cb(ec);
                                   } };
                if (!data)
                {
                    YLOG_CTX_GLOBAL(ctx, info) << "create collector dst_uid=" << draft.dst_uid
                                               << " src_uid=" << draft.src_uid;
                    meta::create_collector(
                        ctx,
                        draft,
                        [uid = draft.dst_uid, cb, this, self](
                            error ec, const collector_info& collector_info) mutable {
                            if (ec) return cb(ec, collector_id());

                            cb(ec, collector_info.id);
                            impl->add_new_streamers(uid, { collector_info });
                        });
                }
                else if (data->collector_info.state == collector_state::deleted)
                {
                    auto meta = make_meta_repo(ctx, data);
                    YLOG_CTX_GLOBAL(ctx, info)
                        << "reset collector "
                        << meta::get_logging_info(data->collector_info).to_string();
                    meta->reset_collector(
                        draft.auth_token,
                        draft.root_folder_id,
                        draft.label_id,
                        std::bind(cb, ph::_1, meta->collector_id()));
                }
                else
                {
                    cb(code::duplicate_collector, data->collector_info.id);
                }
            });
    }

    template <typename Callback>
    void migrate_collector(context_ptr ctx, const migrated_collector_draft& draft, Callback&& cb)
    {
        YLOG_CTX_GLOBAL(ctx, info)
            << "migrate collector dst_uid=" << draft.dst_uid << " src_uid=" << draft.src_uid;
        meta::migrate_collector(
            ctx,
            draft,
            [uid = draft.dst_uid, cb, this, self = shared_from_this()](
                error ec, const collector_info& collector_info) mutable {
                if (ec) return cb(ec, collector_id{});

                cb(ec, collector_info.id);
                impl->add_new_streamers(uid, { collector_info });
            });
    }

    template <typename Callback>
    void unmigrate_collector(
        context_ptr ctx,
        const global_collector_id& id,
        const Callback& user_cb)
    {
        impl->post_operation(
            ctx,
            id,
            operation::edit,
            [id, user_cb](context_ptr ctx, streamer_data_ptr data, const no_data_cb& op_cb) {
                auto cb = multi_cb{ user_cb, op_cb };
                if (data && data->collector_info.state == collector_state::deleted)
                {
                    data = nullptr;
                }
                if (!data) return cb(code::streamer_data_not_found);

                YLOG_CTX_GLOBAL(ctx, info)
                    << "umnigrate collector "
                    << meta::get_logging_info(data->collector_info).to_string();
                auto meta = make_meta_repo(ctx, data);
                meta->update_migration_target_state(
                    meta->state(), data->io->wrap([meta, cb](auto ec) mutable {
                        if (ec) return cb(ec);
                        meta->update_state(collector_state::unmigrated, cb);
                    }));
            });
    }

    template <typename Callback>
    void list_collectors(context_ptr ctx, const uid& uid, Callback&& cb)
    {
        impl->load_user(
            ctx, uid, [cb = std::forward<Callback>(cb)](error ec, user_ptr user) mutable {
                if (ec) return cb(ec, collector_ids{});

                collector_ids res;
                res.reserve(user->streamers.size());
                for (auto& [collector_id, streamer_data] : user->streamers)
                {
                    res.push_back(collector_id);
                }
                return cb(ec, res);
            });
    }

    template <typename Callback>
    void get_collector_info(context_ptr ctx, const global_collector_id& id, Callback&& cb)
    {
        impl->load_user(
            ctx, id.uid, [id, cb = std::forward<Callback>(cb)](error ec, user_ptr user) mutable {
                if (ec) return cb(ec, collector_info());

                auto it = user->streamers.find(id.collector_id);
                if (it == user->streamers.end())
                {
                    cb(code::streamer_data_not_found, collector_info());
                    return;
                }

                auto streamer_data = it->second;
                streamer_data->io->post([cb = std::move(cb), streamer_data]() mutable {
                    auto state = streamer_data->collector_info.state;
                    if (state == collector_state::deleted)
                    {
                        return cb(code::streamer_data_not_found, collector_info());
                    }
                    cb(code::ok, streamer_data->collector_info);
                });
            });
    }

    template <typename Callback>
    void list_users(context_ptr /*ctx*/, Callback&& cb)
    {
        impl->list_users(cb);
    }

    template <typename Callback>
    void unload_user(context_ptr /*ctx*/, const uid& uid, Callback&& cb)
    {
        impl->unload_user(uid, cb);
    }

    void disable_planner()
    {
        planner_->disable();
    }

    void enable_planner()
    {
        planner_->enable();
    }

    module(yplatform::reactor& reactor, const yplatform::ptree& ptree)
    {
        auto planner_conf = std::make_shared<planner_settings>();
        planner_conf->max_concurrency = ptree.get("max_concurrency", 10);
        planner_conf->task_timeout = ptree.get<time_traits::duration>("streaming_timeout");
        planner_conf->task_penalty = ptree.get<time_traits::duration>("streaming_penalty");
        planner_ = std::make_shared<planner_type>(planner_conf, reactor.io());

        auto settings = std::make_shared<streamer_settings>();
        settings->collectors_load_chunk_size = ptree.get("collectors_load_chunk_size", 500);
        settings->message_cache_size = ptree.get("message_cache_size", 100);
        settings->message_chunk_size = ptree.get("message_chunk_size", 10);
        settings->retries_limit = ptree.get("retries_limit", 5);
        settings->skipped_mids_limit = ptree.get("skipped_mids_limit", 100);
        settings->passport_consumer = ptree.get("passport_consumer", "xeno");
        settings->mailbox_settings.internal_api_port = ptree.get("internal_api.port", 5048);
        settings->mailbox_settings.internal_api_service_name =
            ptree.get<std::string>("internal_api.service_name");
        settings->rpop_smtp_data_secret = ptree.get<std::string>("rpop_secret");

        yplatform::read_ptree(settings->allowed_system_labels, ptree, "allowed_system_labels");

        yplatform::read_ptree(settings->allowed_label_types, ptree, "allowed_label_types");

        auto macs = yplatform::find<ymod_macs::module>("macs");

        impl = std::make_shared<streamer_type>(
            reactor,
            planner_,
            std::make_shared<loader_type>(
                reactor.io(), settings->collectors_load_chunk_size, &meta::load_collectors),
            macs,
            settings);
    }

    void init()
    {
        auto shards_distributor =
            yplatform::find<ymod_mdb_sharder::shards_distributor>("shards_distributor");
        shards_distributor->subscribe(
            std::bind(&streamer_type::on_acquire_shards, impl, ph::_1),
            std::bind(&streamer_type::on_release_shards, impl, ph::_1));

        auto users_distributor =
            yplatform::find<ymod_mdb_sharder::users_distributor>("users_distributor");
        users_distributor->set_polling_methods(
            std::bind(&get_all_users, ph::_1, ph::_2),
            std::bind(&get_changed_users, ph::_1, ph::_2, ph::_3, ph::_4));

        users_distributor->subscribe(
            std::bind(&streamer_type::on_acquire_accounts, impl, ph::_1, ph::_2),
            std::bind(&streamer_type::on_release_accounts, impl, ph::_1, ph::_2));
        impl->init();
    }

    void logger(const yplatform::log::source& src)
    { // Shadows base class method.
        impl->logger(src);
    }

    yplatform::ptree get_stats() const
    {
        yplatform::ptree res = impl->calc_stats().get();
        res.put_child("planner", planner_->get_stats());
        return res;
    }

private:
    template <typename Op>
    void post_operation(context_ptr ctx, const uid& dst_uid, const uid& src_uid, Op&& op)
    {
        auto self = yplatform::shared_from(this);
        impl->load_user(
            ctx,
            dst_uid,
            [ctx, src_uid, op = std::forward<Op>(op), this, self](error ec, user_ptr user) mutable {
                auto type = operation::edit;
                auto wrapper = operation_wrapper(ctx, type, std::move(op));
                if (ec)
                {
                    wrapper(ec);
                    return;
                }

                for (auto& [id, data] : user->streamers)
                {
                    if (data->collector_info.src_uid == src_uid)
                    {
                        impl->post_operation(ctx, data->global_id(), type, wrapper);
                        return;
                    }
                }

                wrapper(code::streamer_data_not_found);
            });
    }

    std::shared_ptr<streamer_type> impl;
    planner_ptr planner_;
};

using module_ptr = boost::shared_ptr<module>;

}
