#pragma once

#include "operation.h"

#include <common/json.h>
#include <passport/client.h>

#include <ymod_httpclient/cluster_client.h>
#include <yplatform/find.h>
#include <yplatform/yield.h>

namespace collectors::streamer::operations {

struct finish_migration_op : operation<>
{
    using yield_ctx = yplatform::yield_context<finish_migration_op>;
    using collector_state = collector_state;
    using operation::operation;

    void operator()(yield_ctx ctx)
    {
        reenter(ctx)
        {
            yield passport->get_suid(meta()->dst_uid(), ctx.capture(ec, suid));
            if (ec) yield break;

            yield mark_rpop_migrated(ctx.capture(ec, http_resp));
            if (ec) yield break;
            ec = handle_response(http_resp);
            if (ec) yield break;

            yield meta()->update_state(meta()->migration_target_state(), ctx.capture(ec));
        }
        if (ctx.is_complete()) complete();
    }

    void operator()(yield_ctx::exception_type exception)
    {
        ec = make_error(exception);
        TASK_LOG(context(), error) << "exception during finish_migration_op: " << error_message(ec);
        complete();
    }

    void complete()
    {
        handler(ec);
    }

    error handle_response(const yhttp::response& http_resp)
    {
        error ec = code::ok;
        if (http_resp.status / 100 != 2)
        {
            TASK_LOG(context(), error)
                << "rpop mark_migrated bad http response: " << http_resp.status
                << ", body: " << http_resp.body;
            ec = code::rpop_error;
        }
        else
        {
            auto result = json::from_string(http_resp.body);
            if (result.isMember("error"))
            {
                if (result["error"].isMember("description"))
                {
                    TASK_LOG(context(), error) << "rpop mark_migrated error: "
                                               << result["error"]["description"].asString();
                }
                else
                {
                    TASK_LOG(context(), error)
                        << "rpop mark_migrated bad response: " << http_resp.body;
                }
                ec = code::rpop_error;
            }
        }
        return ec;
    }

    template <typename Handle>
    void mark_rpop_migrated(Handle&& handle)
    {
        auto client = yplatform::find<yhttp::cluster_client>("rpop_client");
        auto req = yhttp::request::GET(
            "/api/mark_migrated?json=1&suid=" + suid +
            "&popid=" + std::to_string(meta()->old_popid()));
        client->async_run(context(), req, std::forward<Handle>(handle));
    }

    passport::client_ptr passport = passport::make_passport_client(context());
    passport::suid suid;
    yhttp::response http_resp;
    error ec;
};

}

#include <yplatform/unyield.h>
