#pragma once

#include "operation.h"

#include <common/json.h>
#include <passport/client.h>

#include <ymod_httpclient/cluster_client.h>
#include <ymod_httpclient/request.h>
#include <yplatform/find.h>
#include <yplatform/yield.h>

namespace collectors::streamer::operations {

struct finish_unmigration_op : operation<>
{
    using yield_ctx = yplatform::yield_context<finish_unmigration_op>;
    using operation::operation;

    void operator()(yield_ctx ctx)
    {
        reenter(ctx)
        {
            if (state()->src_email.empty())
            {
                yield passport()->get_userinfo_by_uid(meta()->src_uid(), ctx.capture(ec, info));
                if (ec)
                {
                    yield break;
                }
                else
                {
                    state()->src_email = info.email;
                }
            }

            yield get_collectors_password(ctx.capture(ec, http_resp));
            if (ec) yield break;
            ec = handle_response(http_resp);
            if (ec) yield break;

            password = resp["smtp_data"]["pass"].asString();

            yield passport()->get_suid(meta()->dst_uid(), ctx.capture(ec, suid));
            if (ec) yield break;

            dst_suid = suid;

            yield reset_migrated_status(ctx.capture(ec, http_resp));
            if (ec) yield break;
            ec = handle_response(http_resp);
            if (ec) yield break;

            if (meta()->migration_target_state() != collector_state::disabled)
            {
                yield enable_old_collector(ctx.capture(ec, http_resp));
                if (ec) yield break;
                ec = handle_response(http_resp);
                if (ec) yield break;
            }

            yield meta()->delete_collector(ctx.capture(ec));
        }
        if (ctx.is_complete()) complete();
    }

    void operator()(yield_ctx::exception_type exception)
    {
        ec = make_error(exception);
        TASK_LOG(context(), error)
            << "exception during finish_unmigration_op: " << error_message(ec);
        complete();
    }

    void complete()
    {
        handler(ec);
    }

    error handle_response(const yhttp::response& http_resp)
    {
        error ec = code::ok;
        if (http_resp.status / 100 != 2)
        {
            TASK_LOG(context(), error)
                << "rpop bad http response: " << http_resp.status << ", body: " << http_resp.body;
            ec = code::rpop_error;
        }
        else
        {
            try
            {
                auto result = json::from_string(http_resp.body);
                if (result.isMember("error"))
                {
                    if (result["error"].isMember("description"))
                    {
                        TASK_LOG(context(), error)
                            << "rpop error: " << result["error"]["description"].asString();
                    }
                    else
                    {
                        TASK_LOG(context(), error) << "rpop bad response: " << http_resp.body;
                    }
                    ec = code::rpop_error;
                }
                else
                {
                    resp = result;
                }
            }
            catch (const std::exception& e)
            {
                TASK_LOG(context(), error) << "parse rpop response exception: " << e.what();
                ec = code::operation_exception;
            }
        }
        return ec;
    }

    template <typename Handle>
    void get_collectors_password(Handle&& handle)
    {
        auto client = yplatform::find<yhttp::cluster_client>("rpop_client");
        auto req = yhttp::request::POST(
            "/api/v2/smtp_data",
            yhttp::form_encode({ { "uid", meta()->dst_uid() },
                                 { "email", state()->src_email },
                                 { "secret", settings()->rpop_smtp_data_secret } }));
        client->async_run(context(), req, std::forward<Handle>(handle));
    }

    template <typename Handle>
    void reset_migrated_status(Handle&& handle)
    {
        auto client = yplatform::find<yhttp::cluster_client>("rpop_client");
        auto req = yhttp::request::POST(
            "/api/edit",
            yhttp::form_encode({ { "json", 1 },
                                 { "mdb", "pg" },
                                 { "suid", dst_suid },
                                 { "popid", meta()->old_popid() },
                                 { "user", state()->dst_email },
                                 { "password", password } }));
        client->async_run(context(), req, std::forward<Handle>(handle));
    }

    template <typename Handle>
    void enable_old_collector(Handle&& handle)
    {
        auto client = yplatform::find<yhttp::cluster_client>("rpop_client");
        auto req = yhttp::request::POST(
            "/api/enable",
            yhttp::form_encode({ { "json", 1 },
                                 { "mdb", "pg" },
                                 { "suid", dst_suid },
                                 { "popid", meta()->old_popid() },
                                 { "is_on", 1 } }));
        client->async_run(context(), req, std::forward<Handle>(handle));
    }

    json::value resp;
    std::string password;
    std::string dst_suid;
    passport::suid suid;
    user_info info;
    yhttp::response http_resp;
    error ec;
};

}

#include <yplatform/unyield.h>
