#pragma once

#include "operation.h"
#include "finish_migration_op.h"
#include "finish_unmigration_op.h"
#include "map_folders_ignoring_folder_struct_op.h"
#include "map_folders_reproducing_folder_struct_op.h"
#include "sync_messages_op.h"

#include <common/typed_log.h>

#include <yplatform/yield.h>

namespace collectors::streamer::operations {

template <typename Handler>
std::function<void(void)> get_folders_mapping_op(detail::environment_ptr env, Handler&& h)
{
    if (env->meta->ignore_folders_struct())
    {
        return [env, h]() {
            auto op = std::make_shared<map_folders_ignoring_folder_struct_op>(env, h);
            yplatform::spawn(env->io->get_executor(), op);
        };
    }
    else
    {
        return [env, h]() {
            auto op = std::make_shared<map_folders_reproducing_folder_struct_op>(env, h);
            yplatform::spawn(env->io->get_executor(), op);
        };
    }
}

struct main_op : operation<>
{
    using yield_ctx = yplatform::yield_context<main_op>;
    using collector_state = collector_state;
    using operation::operation;

    std::uint32_t prev_sent_count = 0;
    user_info user_info;
    error ec;

    void operator()(yield_ctx ctx)
    {
        reenter(ctx)
        {
            TASK_LOG(context(), info)
                << "starting iteration streamer_state=" << to_string(meta()->state());
            meta()->update_last_run_ts(std::time(nullptr));
            prev_sent_count = state()->sent_count;

            if (state()->dst_email.empty())
            {
                yield passport()->get_userinfo_by_uid(
                    meta()->dst_uid(), ctx.capture(ec, user_info));
                if (ec) yield break;

                state()->dst_email = user_info.email;
            }

            if (meta()->state() == collector_state::migrated)
            {
                yield spawn<finish_migration_op>(ctx.capture(ec));
                if (ec) yield break;
            }
            if (meta()->state() == collector_state::unmigrated)
            {
                yield spawn<finish_unmigration_op>(ctx.capture(ec));
                if (!ec)
                {
                    ec = code::deleted_collector;
                }
                yield break;
            }

            if (meta()->state() == collector_state::disabled) yield break;
            if (meta()->state() == collector_state::deleted)
            {
                if (state()->src_email.empty())
                {
                    yield passport()->get_userinfo_by_uid(
                        meta()->src_uid(), ctx.capture(ec, user_info));
                    if (ec && ec != code::user_not_found) yield break;

                    state()->src_email = user_info.email;
                }

                if (can_remove_alias())
                {
                    TASK_LOG(context(), info) << "removing alias";
                    yield passport()->remove_alias(
                        meta()->dst_uid(),
                        state()->src_email,
                        settings()->passport_consumer,
                        ctx.capture(ec));
                    if (ec) yield break;
                }

                TASK_LOG(context(), info) << "deleting collector";
                yield meta()->delete_collector(ctx.capture(ec));
                if (!ec)
                {
                    ec = code::deleted_collector;
                }
                yield break;
            }

            yield check_auth_token(ctx.capture(ec, user_info));
            if (ec)
            {
                if (ec == code::invalid_auth_token)
                {
                    TASK_LOG(context(), error) << "resetting invalid auth token";
                    yield meta()->reset_token(ctx.capture(ec)); // XXX reset_auth_token()?
                }
                yield break;
            }

            if (user_info.uid != meta()->src_uid())
            {
                TASK_LOG(context(), error) << "token for wrong uid";
                ec = { code::invalid_auth_token, "token for wrong uid: " + user_info.uid };
                yield break;
            }

            state()->src_email = user_info.email;
            state()->rpop_info =
                make_rpop_info(state()->get_src_login(), meta()->original_server());

            if (meta()->state() == collector_state::created)
            {
                TASK_LOG(context(), info) << "adding alias";
                yield passport()->add_alias(
                    meta()->dst_uid(),
                    state()->src_email,
                    settings()->passport_consumer,
                    ctx.capture(ec));
                if (!ec)
                {
                    yield meta()->update_state(
                        collector_state::ready, ctx.capture(ec)); // XXX set_state()?
                    if (ec) yield break;
                }
                else
                {
                    TASK_LOG(context(), error) << "add alias error: " << ec.message();
                }
            }

            yield get_folders_mapping_op(env, ctx.capture(ec))();
            if (ec) yield break;

            yield spawn<sync_messages_op>(ctx.capture(ec));
            if (ec) yield break;
        }
        if (ctx.is_complete())
        {
            TASK_LOG(context(), info)
                << "finishing iteration streamer_state=" << to_string(meta()->state())
                << " error_status=" << error_message(ec);
            typed::log_iteration(
                context(),
                get_logging_info(meta()),
                meta()->state(),
                meta()->last_run_ts(),
                state()->sent_count - prev_sent_count,
                state()->last_mid,
                ec);
            complete();
        }
    }

    void operator()(yield_ctx::exception_type exception)
    {
        ec = make_error(exception);
        TASK_LOG(context(), error) << "exception during main_op: " << error_message(ec);
        complete();
    }

    void complete()
    {
        handler(ec);
    }

    template <typename Handler>
    void check_auth_token(Handler&& handler)
    {
        if (meta()->auth_token().empty())
        {
            return handler(code::no_auth_token, collectors::user_info{});
        }
        passport()->check_auth_token(meta()->auth_token(), {}, std::forward<Handler>(handler));
    }

    std::string make_rpop_info(const std::string& src_login, const std::string& original_server)
    {
        return src_login + "@" + original_server;
    }

    bool can_remove_alias()
    {
        return state()->src_email.size();
    }
};

}

#include <yplatform/unyield.h>
