#pragma once

#include "operation.h"
#include "get_or_create_folder_op.h"

#include <yplatform/yield.h>

namespace collectors::streamer::operations {

inline bool compare_folders_by_parent_fid(const folder& f, const folder& s)
{
    return f.parent_fid < s.parent_fid;
}

// sorts all src_folders by parent_id
// process all folders with 0 parent_id
// iterate over just processed folder, and process all childs for them (find using lower_bound)
// repeat untill nothing to process
// total complexity nlog(n)

struct map_folders_reproducing_folder_struct_op : operation<>
{
    using yield_ctx = yplatform::yield_context<map_folders_reproducing_folder_struct_op>;
    using operation::operation;

    void operator()(yield_ctx ctx)
    {
        reenter(ctx)
        {
            TASK_LOG(context(), info) << "syncing folders";
            state()->clear_folders();
            state()->folders_mapping[EMPTY_ID] = meta()->root_folder_id();

            yield src_mailbox()->get_folders(ctx.capture(ec, folders));
            if (ec) yield break;

            src_folders = folders;
            std::sort(src_folders.begin(), src_folders.end(), compare_folders_by_parent_fid);
            fids_to_search = { EMPTY_ID };
            while (fids_to_search.size())
            {
                found_fids.clear();
                for (fid_it = fids_to_search.begin(); fid_it != fids_to_search.end(); ++fid_it)
                {
                    src_folder_it = std::lower_bound(
                        src_folders.begin(),
                        src_folders.end(),
                        *fid_it,
                        [](const folder& folder, const fid& parent_fid) {
                            return folder.parent_fid < parent_fid;
                        });

                    while (src_folder_it != src_folders.end() &&
                           src_folder_it->parent_fid == *fid_it)
                    {
                        state()->cached_src_folders[src_folder_it->fid] = *src_folder_it;
                        if (!is_empty_id(meta()->root_folder_id()))
                        {
                            src_folder_it->symbol.clear();
                        }

                        yield spawn<get_or_create_folder_op>(
                            ctx.capture(ec, dst_folder), *src_folder_it);
                        if (ec) yield break;

                        state()->folders_mapping[src_folder_it->fid] = dst_folder.fid;
                        found_fids.insert(src_folder_it->fid);
                        ++src_folder_it;
                    }
                }
                fids_to_search = found_fids;
            }
        }
        if (ctx.is_complete()) complete();
    }

    void operator()(yield_ctx::exception_type exception)
    {
        ec = make_error(exception);
        TASK_LOG(context(), error)
            << "exception during map_folders_reproducing_folder_struct_op: " << error_message(ec);
        complete();
    }

    void complete()
    {
        handler(ec);
    }

    folders src_folders;
    folder dst_folder;
    folders::iterator src_folder_it;
    std::set<fid> fids_to_search;
    std::set<fid>::iterator fid_it;
    std::set<fid> found_fids;
    folders folders;
    error ec;
};

}
