#pragma once

#include <common/types.h>
#include <mailbox/mailbox.h>
#include <streamer/settings.h>
#include <streamer/streamer_data.h>
#include <streamer/meta.h>
#include <passport/client.h>

#include <yplatform/coroutine.h>

namespace collectors::streamer::operations {

namespace detail {

// XXX why environment_S_.h?

struct environment
{
    context_ptr context;
    mailbox::read_only_mailbox_ptr src_mailbox;
    mailbox::mailbox_ptr dst_mailbox;
    meta::repository_ptr meta;
    passport::client_ptr passport;
    sync_state_ptr state;
    streamer_settings_ptr settings;
    boost::asio::io_service* io;
};

using environment_ptr = std::shared_ptr<environment>;

}

template <typename Handler = no_data_cb>
struct operation
{
    operation(detail::environment_ptr env, Handler handler) : handler(std::move(handler)), env(env)
    {
    }

protected:
    auto context()
    {
        return env->context;
    } // XXX ctx()?
    auto src_mailbox()
    {
        if (!env->src_mailbox) throw std::runtime_error("empty src_mailbox");
        return env->src_mailbox;
    }
    auto dst_mailbox()
    {
        return env->dst_mailbox;
    }
    auto meta()
    {
        return env->meta;
    }
    auto passport()
    {
        return env->passport;
    }
    auto state()
    {
        return env->state;
    }
    auto settings()
    {
        return env->settings;
    }

    template <typename Op, typename OpHandler, typename... Args>
    void spawn(OpHandler&& h, Args&&... args)
    {
        auto op = std::make_shared<Op>(env, std::forward<OpHandler>(h));
        yplatform::spawn(env->io->get_executor(), op, std::forward<Args>(args)...);
    }

    std::decay_t<Handler> handler;
    detail::environment_ptr env;
};

template <typename Op, typename... Args>
inline void spawn(
    context_ptr ctx,
    streamer_data_ptr streamer_data,
    streamer_settings_ptr settings,
    const no_data_cb& handler,
    Args&&... args)
{
    auto env = std::make_shared<detail::environment>();
    env->context = ctx;
    env->meta = make_meta_repo(ctx, streamer_data);
    env->passport = passport::make_passport_client(ctx);
    env->state = std::shared_ptr<sync_state>(streamer_data, &streamer_data->sync_state);
    env->io = streamer_data->io;
    env->settings = settings;
    env->dst_mailbox = mailbox::make_mailbox(ctx, streamer_data->collector_info.dst_uid);

    auto spawn_op = [env, tuple_args = std::tuple{ args... }, handler](auto&& ec, auto&& mailbox) {
        if (ec == code::user_not_found)
        {
            env->src_mailbox = nullptr;
        }
        else if (ec)
        {
            return handler(ec);
        }
        else
        {
            env->src_mailbox = mailbox;
        }

        auto op = std::make_shared<Op>(env, handler);
        std::apply(
            [](auto... args) { yplatform::spawn(args...); },
            std::tuple_cat(std::tuple{ env->io->get_executor(), op }, tuple_args));
    };

    if (streamer_data->collector_info.ignore_folders_struct)
    {
        mailbox::make_read_only_pop3_mailbox(
            ctx,
            streamer_data->collector_info.src_uid,
            std::shared_ptr<mailbox::settings>(settings, &settings->mailbox_settings),
            spawn_op);
    }
    else
    {
        mailbox::make_read_only_mailbox(
            ctx,
            streamer_data->collector_info.src_uid,
            std::shared_ptr<mailbox::settings>(settings, &settings->mailbox_settings),
            spawn_op);
    }
}

}
