#pragma once

#include "operation.h"
#include "sync_labels_op.h"

#include <common/typed_log.h>

#include <yplatform/yield.h>

namespace collectors::streamer::operations {

struct sync_messages_op : operation<>
{
    using yield_ctx = yplatform::yield_context<sync_messages_op>;
    using operation::operation;

    void operator()(yield_ctx ctx)
    {
        reenter(ctx)
        {
            TASK_LOG(context(), info) << "syncing messages last_mid=" << state()->last_mid;
            while (stores_count < settings()->message_chunk_size)
            {
                yield get_first_message(ctx.capture(ec, src_message));
                if (ec == code::no_more_messages)
                {
                    ec = {};
                    break;
                }
                if (ec) yield break;

                if (need_sync(src_message))
                {
                    yield sync_labels_for_message(ctx.capture(ec), src_message);
                    if (ec) break;

                    dst_message = map_message(src_message);
                    yield store_message(ctx.capture(ec, dst_message.mid), dst_message);
                    typed::log_store_message(
                        context(),
                        meta()->global_id(),
                        { meta()->src_uid(), state()->src_email },
                        { meta()->dst_uid(), state()->dst_email },
                        src_message,
                        dst_message,
                        ec);
                    if (ec && ec != code::message_is_duplicate)
                    {
                        if (++state()->last_message_retries > settings()->retries_limit)
                        {
                            yield add_message_to_skipped(ec, src_message, ctx.capture(ec));
                        }
                        else
                        {
                            process_error();
                        }

                        if (ec) break;
                    }
                    if (!ec) ++state()->sent_count;
                }

                update_last_mid(src_message.mid);
                advance_to_next_message();
            }

            if (sync_has_progress())
            {
                yield save_last_mid(ctx.capture(ec));
                if (ec) yield break;
            }
        }
        if (ctx.is_complete()) complete();
    }

    void operator()(yield_ctx::exception_type exception)
    {
        ec = make_error(exception);
        TASK_LOG(context(), error) << "exception during sync_messages_op: " << error_message(ec);
        complete();
    }

    void complete()
    {
        handler(ec);
    }

    template <typename Handler>
    void get_first_message(Handler&& handler)
    {
        if (state()->cached_messages.empty())
        {
            TASK_LOG(context(), info) << "fetching new chunk";
            src_mailbox()->get_next_message_chunk(
                state()->last_mid,
                settings()->message_cache_size,
                [handler = std::forward<Handler>(handler),
                 this](error ec, const messages& messages) mutable {
                    if (ec) return handler(ec, message());

                    state()->cached_messages = messages;
                    std::reverse(state()->cached_messages.begin(), state()->cached_messages.end());
                    get_last_message_from_cache(handler);
                });
        }
        else
        {
            get_last_message_from_cache(handler);
        }
    }

    bool need_sync(const message& msg)
    {
        auto it = state()->cached_src_folders.find(msg.fid);
        if (it == state()->cached_src_folders.end())
        {
            throw std::runtime_error("can't find cached src folder, fid=" + msg.fid);
        }
        return !it->second.skip_messages;
    }

    template <typename Handler>
    void get_last_message_from_cache(Handler&& handler)
    {
        if (state()->cached_messages.empty())
        {
            return handler(code::no_more_messages, message());
        }

        const auto& res = state()->cached_messages.back();
        if (state()->last_message.mid != res.mid)
        {
            state()->last_message = res;
            state()->last_message_retries = 0;
        }
        handler(code::ok, res);
    }

    void advance_to_next_message()
    {
        state()->cached_messages.pop_back();
    }

    labels map_labels(const labels& src_labels)
    {
        labels dst_labels;
        for (auto label : src_labels)
        {
            auto it = state()->labels_mapping.find(label.lid);
            if (it != state()->labels_mapping.end())
            {
                label.lid = it->second;
                dst_labels.push_back(label);
            }
        }
        return dst_labels;
    }

    fid map_fid(const fid& fid)
    {
        auto it = state()->folders_mapping.find(fid);
        if (it != state()->folders_mapping.end())
        {
            return it->second;
        }
        throw std::runtime_error("can't map folder, fid=" + fid);
    }

    template <typename Handler>
    void sync_labels_for_message(Handler&& handler, const message& msg)
    {
        spawn<sync_labels_op>(handler, msg.labels);
    }

    message map_message(const message& msg)
    {
        message res = msg;
        res.labels = map_labels(msg.labels);
        res.fid = map_fid(msg.fid);
        return res;
    }

    template <typename Handler>
    void store_message(Handler&& handler, const message& message)
    {
        ++stores_count;
        bool old_message = message.date < meta()->creation_ts();
        dst_mailbox()->store_message(
            message,
            state()->dst_email,
            old_message, // disable_push
            old_message, // skip_loop_prevention
            state()->rpop_ids,
            state()->rpop_info,
            handler);
    }

    template <typename Handler>
    void add_message_to_skipped(error ec, const message& msg, Handler&& handler)
    {
        TASK_LOG(context(), error)
            << "skipping message src_mid=" << msg.mid << " last_error=" << ec.message();
        auto skipped_mids = meta()->skipped_mids();
        skipped_mids.push_back(msg.mid);
        meta()->update_skipped_mids(skipped_mids, handler);
    }

    void process_error()
    {
        // Could be any reason: folder/label deleted, src message deleted
        state()->clear_cache();
    }

    bool sync_has_progress()
    {
        return state()->last_mid != meta()->last_mid();
    }

    void update_last_mid(const mid& mid)
    {
        state()->last_mid = mid;
    }

    template <typename Handler>
    void save_last_mid(Handler&& handler)
    {
        meta()->update_last_mid(state()->last_mid, handler);
    }

    std::uint32_t stores_count = 0;
    message src_message;
    message dst_message;
    error ec;
};

}

#include <yplatform/unyield.h>
