#pragma once

#include "settings.h"

#include <common/types.h>

#include <yplatform/future/future.hpp>

#include <memory>

namespace collectors::streamer {

namespace ph = std::placeholders;

using task_function = std::function<void(context_ptr, const no_data_cb&)>;

namespace detail {

template <typename ExternalId>
struct task
{
    enum class queued_state
    {
        idle = 0,
        running,
        removed
    };

    ExternalId id;
    task_function fun;
    time_traits::time_point last_run_ts = time_traits::time_point::min();
    queued_state state = queued_state::idle;
    context_ptr last_ctx = {};

    bool operator<(const task<ExternalId>& s) const
    {
        return std::tie(state, last_run_ts) < std::tie(s.state, s.last_run_ts);
    }
};

template <typename Index, typename Element>
class indexed_multiset
{
    using queue_type = std::multiset<Element>;
    using elements_map = std::map<Index, typename queue_type::iterator>;

public:
    void insert(const Index& i, const Element& e)
    {
        if (elements_.find(i) != elements_.end())
        {
            throw std::runtime_error("element index already exists");
        }

        auto it = queue_.insert(e);
        elements_[i] = it;
    }

    void erase(const Index& i)
    {
        auto elements_it = elements_.find(i);
        if (elements_it == elements_.end())
        {
            throw std::runtime_error("no such index");
        }

        queue_.erase(elements_it->second);
        elements_.erase(elements_it);
    }

    void update(const Index& i, const Element& e)
    {
        auto elements_it = elements_.find(i);
        if (elements_it == elements_.end())
        {
            throw std::runtime_error("no such index");
        }
        queue_.erase(elements_it->second);
        elements_it->second = queue_.insert(e);
    }

    std::size_t size()
    {
        return elements_.size();
    }

    const Element& first()
    {
        return *queue_.begin();
    }

    auto begin()
    {
        return queue_.begin();
    }
    auto end()
    {
        return queue_.end();
    }
    auto find(const Index& i)
    {
        auto elements_it = elements_.find(i);
        if (elements_it == elements_.end())
        {
            return end();
        }

        return elements_it->second;
    }

private:
    queue_type queue_;
    elements_map elements_;
};

}

template <typename TaskId>
class planner : public std::enable_shared_from_this<planner<TaskId>>
{
    using timer = time_traits::timer;
    using timer_ptr = time_traits::timer_ptr; // XXX aliase time_traits::* in types.h?
    using queued_task = detail::task<TaskId>;
    using queue = detail::indexed_multiset<TaskId, queued_task>;

public:
    using remove_handler_type = std::function<void(const TaskId&)>;

    planner(planner_settings_ptr settings, boost::asio::io_context* io)
        : settings_(settings), io_(io)
    {
    }

    void set_remove_handler(const remove_handler_type& h)
    {
        remove_handler_ = h;
    }

    void add(const TaskId& id, const task_function& fun)
    {
        auto self = this->shared_from_this();
        io_->post([id, fun, this, self]() {
            auto it = queue_.find(id);
            if (it == queue_.end())
            {
                queue_.insert(id, queued_task{ id, fun });
            }
            else
            {
                queue_.update(id, queued_task{ id, fun });
            }

            run_availiable_tasks();
        });
    }

    void remove(const TaskId& id)
    {
        auto self = this->shared_from_this();
        io_->post([id, this, self]() {
            auto it = queue_.find(id);
            if (it == queue_.end())
            {
                return;
            }

            auto task = *it;
            if (task.state != queued_task::queued_state::idle)
            {
                task.last_ctx->cancel();
                task.state = queued_task::queued_state::removed;
                queue_.update(id, task);
            }
            else
            {
                remove_task(id);
            }
        });
    }

    void stop(const TaskId& id)
    {
        auto it = queue_.find(id);
        if (it == queue_.end())
        {
            return;
        }

        auto task = *it;
        if (task.state == queued_task::queued_state::running)
        {
            task.last_ctx->cancel();
        }
    }

    void disable()
    {
        auto self = this->shared_from_this();
        io_->post([this, self]() { disabled_ = true; });
    }

    void enable()
    {
        auto self = this->shared_from_this();
        io_->post([this, self]() {
            disabled_ = false;
            run_availiable_tasks();
        });
    }

    yplatform::ptree get_stats()
    {
        yplatform::ptree res;
        res.put_child("delays_hgram", calc_delays().get());
        return res;
    }

    // XXX how to reset queue or drop single streamer?

private:
    void run_availiable_tasks()
    {
        if (disabled_) return;

        if (timer_)
        {
            timer_.reset(); // XXX cancel?
        }

        auto now = time_traits::clock::now();
        while (queue_.size() > running_count_ && running_count_ < settings_->max_concurrency)
        {
            auto& queued = queue_.first();
            auto next_run_ts = queued.last_run_ts + settings_->task_penalty;

            if (now < next_run_ts)
            {
                auto self = this->shared_from_this();
                timer_ = std::make_shared<timer>(*io_, next_run_ts);
                timer_->async_wait([this, self](auto ec) {
                    if (ec != boost::asio::error::operation_aborted)
                    {
                        run_availiable_tasks();
                    }
                });
                return;
            }
            if (queued.state == queued_task::queued_state::idle)
            {
                start_task(queued);
            }
            else
            {
                break; // sorted by state, nothing left
            }
        }
    }

    void start_task(queued_task task)
    {
        auto ctx = boost::make_shared<yplatform::task_context>(); // XXX reuse the original ctx?
        task.state = queued_task::queued_state::running;
        task.last_ctx = ctx;
        queue_.update(task.id, task);
        ++running_count_;

        ctx->deadline_from_now(settings_->task_timeout);
        auto self = this->shared_from_this();
        task.fun(ctx, [id = task.id, this, self](error ec) { handle_task_finish(ec, id); });
    }

    // XXX replace post by io.wrap on task call?
    void handle_task_finish(error ec, const TaskId& id)
    {
        auto self = this->shared_from_this();
        io_->post([ec, id, this, self]() mutable {
            auto it = queue_.find(id);
            if (it == queue_.end())
            {
                return;
            }

            auto task = *it;
            if (task.state == queued_task::queued_state::removed || ec == errc::task_expired)
            {
                remove_task(id);
            }
            else
            {
                task.last_run_ts = time_traits::clock::now();
                task.state = queued_task::queued_state::idle;
                queue_.update(id, task);
            }

            --running_count_;
            run_availiable_tasks();
        });
    }

    void remove_task(const TaskId& id)
    {
        queue_.erase(id);
        if (remove_handler_)
        {
            remove_handler_(id);
        }
    }

    yplatform::future::future<yplatform::ptree> calc_delays()
    {
        yplatform::future::promise<yplatform::ptree> prom;
        auto self = this->shared_from_this();
        io_->post([prom, this, self]() mutable {
            yplatform::ptree res;
            auto now = yplatform::time_traits::clock::now();
            for (auto& task : queue_)
            {
                if (task.last_run_ts != time_traits::time_point::min())
                {
                    yplatform::ptree item;
                    item.put("", time_traits::to_string(now - task.last_run_ts));
                    res.push_back(std::pair{ "", item });
                }
            }
            prom.set(res);
        });
        return prom;
    }

    planner_settings_ptr settings_;
    boost::asio::io_context* io_;
    timer_ptr timer_;
    queue queue_;
    std::size_t running_count_ = 0;
    remove_handler_type remove_handler_;

    bool disabled_ = false;
};

}
