#pragma once

#include "sync_state.h"

#include <memory>

namespace collectors::streamer {

struct streamer_data;
using streamer_data_ptr = std::shared_ptr<streamer_data>;
using user_operation = std::function<void(context_ptr, streamer_data_ptr, const no_data_cb&)>;

enum class operation
{
    noop,
    edit,
    exec
};

class operation_wrapper
{
public:
    template <typename Op>
    operation_wrapper(context_ptr ctx, operation type, Op&& op)
        : ctx(ctx), type(type), op(std::forward<Op>(op))
    {
    }

    template <>
    operation_wrapper(context_ptr ctx, operation type, operation_wrapper&& wrapper)
        : ctx(ctx), type(type), op(std::move(wrapper.op))
    {
    }

    void operator()(streamer_data_ptr data, const no_data_cb& cb)
    {
        op(ctx, data, cb);
    }
    void operator()(error /*ec*/)
    {
        op(ctx, {}, {});
    }

    operation get_type()
    {
        return type;
    }

private:
    context_ptr ctx;
    operation type;
    user_operation op;
};

struct streamer_data
{
    streamer_data(boost::asio::io_context* io, const collector_info& collector_info)
        : io(io)
        , collector_info(collector_info)
        , sync_state(collector_info.last_mid, make_rpop_ids())
    {
    }

    boost::asio::io_context* io;
    operation op = operation::noop;

    collector_info collector_info;
    sync_state sync_state;
    bool planned = false;

    std::deque<operation_wrapper> operations_queue;

    global_collector_id global_id()
    {
        return global_collector_id(collector_info.dst_uid, collector_info.id);
    }

    std::vector<std::string> make_rpop_ids()
    {
        std::vector<std::string> res{ global_id().to_string() };
        if (collector_info.old_popid)
        {
            res.emplace_back(std::to_string(collector_info.old_popid));
        }
        return res;
    }
};

enum class user_state
{
    initial,
    api_ready,
    full_ready,
    released
};

struct user
{
    uid uid;
    shard_id shard;
    user_state state = user_state::initial;
    bool loading = false;

    std::map<collector_id, streamer_data_ptr> streamers = {};
};

using user_ptr = std::shared_ptr<user>;
using user_cb = std::function<void(error, user_ptr)>;

}
