#pragma once

#include "streamer_data.h"
#include "settings.h"
#include "user_polling.h"
#include "meta/repository.h"
#include "operations/main_op.h"

#include <mail_errors/error_code.h>
#include <yplatform/find.h>
#include <yplatform/future/future.hpp>
#include <yplatform/log/contains_logger.h>

namespace collectors::streamer::detail {

template <typename PlannerPtr, typename LoaderPtr, typename MacsModulePtr>
class streamer_impl
    : public std::enable_shared_from_this<streamer_impl<PlannerPtr, LoaderPtr, MacsModulePtr>>
    , public yplatform::log::contains_logger
{
public:
    streamer_impl(
        yplatform::reactor& reactor,
        PlannerPtr planner,
        LoaderPtr loader,
        MacsModulePtr macs,
        streamer_settings_ptr settings)
        : reactor_(reactor)
        , io_(reactor.io())
        , planner_(planner)
        , loader_(loader)
        , macs_(macs)
        , settings_(settings)
    {
    }

    void init()
    {
        auto self = yplatform::shared_from(this);
        loader_->set_handler(
            io_->wrap(std::bind(&streamer_impl::handle_collectors, self, ph::_1, ph::_2)));
        planner_->set_remove_handler(
            io_->wrap(std::bind(&streamer_impl::handle_planner_remove, self, ph::_1)));
    }

    void on_acquire_shards(const std::vector<shard_id>& /*shard_ids*/)
    {
        // Do nothing.
    }

    void on_acquire_accounts(const shard_id& shard_id, const std::vector<uint64_t>& acquired_uids)
    {
        auto self = yplatform::shared_from(this);
        io_->post([shard_id, acquired_uids, this, self]() {
            uids to_load;
            for (auto& elem : acquired_uids)
            {
                auto uid = std::to_string(elem);
                if (!update_acquired_user(shard_id, uid))
                {
                    users_.insert({ uid, std::make_shared<user>(user{ uid, shard_id }) });
                    to_load.push_back(uid);
                }
            }
            if (to_load.size())
            {
                loader_->process(shard_id, to_load);
            }
        });
    }

    void on_release_shards(const std::vector<shard_id>& shards)
    {
        auto self = yplatform::shared_from(this);
        io_->post([shards, this, self]() {
            std::vector<user_ptr> released_users;
            for (auto& shard : shards)
            {
                for (auto& [uid, user] : users_)
                {
                    if (user->shard == shard)
                    {
                        released_users.push_back(user);
                    }
                }
                loader_->cancel(shard);
            }

            for (auto& user : released_users)
            {
                try_release_user_streamers(user);
            }
        });
    }

    void on_release_accounts(const std::string& shard_id, const std::vector<uint64_t>& uids)
    {
        auto self = yplatform::shared_from(this);
        io_->post([shard_id, uids, this, self]() {
            for (auto elem : uids)
            {
                auto uid = std::to_string(elem);
                auto it = users_.find(uid);
                if (it != users_.end())
                {
                    auto& user = it->second;
                    if (user->shard == shard_id)
                    {
                        try_release_user_streamers(user);
                    }
                }
            }
        });
    }

    void add_new_streamers(const uid& uid, const collector_info_chunk& chunk)
    {
        auto self = yplatform::shared_from(this);
        io_->post([uid, chunk, this, self]() {
            auto it = users_.find(uid);
            if (it == users_.end())
            {
                return;
            }

            add_streamers(it->second, chunk);
        });
    }

    template <typename Handler>
    void get_user(const uid& uid, Handler&& h)
    {
        auto self = yplatform::shared_from(this);
        io_->post([uid, h = std::forward<Handler>(h), this, self]() mutable {
            auto user = find_user(uid);
            if (user)
            {
                h(code::ok, user);
            }
            else
            {
                h(code::user_not_found, nullptr);
            }
        });
    }

    template <typename Handler>
    void load_user(context_ptr ctx, const uid& uid, Handler&& h)
    {
        auto self = yplatform::shared_from(this);
        get_user(
            uid,
            [ctx, uid, h = std::forward<Handler>(h), this, self](error ec, user_ptr user) mutable {
                if (!ec && user->state != user_state::initial)
                {
                    return h(
                        user->state == user_state::released ? code::user_released : code::ok, user);
                }

                macs_->get_user_shard_id(
                    ctx,
                    uid,
                    io_->wrap([uid, h = std::move(h), this, self](
                                  mail_errors::error_code ec, const shard_id& shard) mutable {
                        if (ec)
                        {
                            YLOG_L(error) << "load user, cannot get user shard: " << ec.message();
                            h(ec, {});
                            return;
                        }

                        load_handlers_[uid].push_back(std::move(h));
                        loader_->process_prior(shard, { uid });
                    }));
            });
    }

    template <typename Op>
    void post_operation(context_ptr ctx, const global_collector_id& id, operation type, Op&& op)
    {
        auto self = yplatform::shared_from(this);
        load_user(
            ctx,
            id.uid,
            [ctx, id, type, op = std::forward<Op>(op), this, self](
                error ec, user_ptr user) mutable {
                auto wrapper = operation_wrapper(ctx, type, std::move(op));
                if (ec)
                {
                    wrapper(ec);
                    return;
                }

                auto it = user->streamers.find(id.collector_id);
                if (it == user->streamers.end())
                {
                    wrapper(code::streamer_data_not_found);
                    return;
                }

                auto streamer = it->second;
                streamer->operations_queue.emplace_back(std::move(wrapper));
                process_streamer_queue(streamer);
            });
    }

    template <typename Callback>
    void list_users(Callback&& cb)
    {
        auto self = yplatform::shared_from(this);
        io_->post([cb = std::forward<Callback>(cb), this, self]() mutable {
            uids res;
            res.reserve(users_.size());
            for (auto& [uid, user] : users_)
            {
                if (user->streamers.size() && user->state == user_state::full_ready)
                {
                    res.push_back(uid);
                }
            }
            cb(code::ok, res);
        });
    }

    template <typename Callback>
    void unload_user(const uid& uid, Callback&& cb)
    {
        auto self = yplatform::shared_from(this);
        io_->post([cb = std::forward<Callback>(cb), uid, this, self]() mutable {
            get_user(uid, [cb, this, self](error ec, auto&& user) {
                if (ec) return cb(ec);

                try_release_user_streamers(user);
                cb(code::ok);
            });
        });
    }

    yplatform::future::future<yplatform::ptree> calc_stats() const
    {
        yplatform::future::promise<yplatform::ptree> prom;
        auto self = yplatform::shared_from(this);
        io_->post([prom, this, self]() mutable {
            yplatform::ptree res;
            size_t users_count = 0;
            size_t collectors_count = 0;
            for (auto& [uid, user] : users_)
            {
                if (user->state != user_state::full_ready) continue;
                users_count++;
                collectors_count += user->streamers.size();
            }
            res.put("users_count", users_count);
            res.put("collectors_count", collectors_count);
            prom.set(res);
        });
        return prom;
    }

private:
    void handle_collectors(const shard_id shard, const loaded_users& loaded)
    {
        YLOG_L(debug) << "loaded users chunk size=" << loaded.size();
        for (auto& [uid, chunk] : loaded)
        {
            auto user = find_user(uid);
            auto handlers_it = load_handlers_.find(uid);
            if (!user)
            {
                if (handlers_it == load_handlers_.end())
                {
                    continue;
                }

                user = std::make_shared<streamer::user>(
                    streamer::user{ uid, shard, user_state::api_ready });
                users_.insert({ uid, user });
            }
            else if (user->state == user_state::initial)
            {
                user->state = user_state::full_ready;
            }

            add_streamers(user, chunk);
            if (handlers_it != load_handlers_.end())
            {
                for (auto& handler : handlers_it->second)
                {
                    try
                    {
                        handler({}, user);
                    }
                    catch (const std::exception& e)
                    {
                        YLOG_L(error) << "load handler exception: " << e.what();
                    }
                }
                load_handlers_.erase(handlers_it);
            }
        }
    }

    void handle_op_finish(const global_collector_id& id)
    {
        auto user = find_user(id.uid);
        if (!user) return;

        auto it = user->streamers.find(id.collector_id);
        if (it == user->streamers.end()) return;

        it->second->op = operation::noop;
        if (user->state == user_state::released || user->state == user_state::initial)
        {
            try_release_streamer(user, it->second);
        }
        else
        {
            process_streamer_queue(it->second);
        }
    }

    void handle_planner_remove(const global_collector_id& id)
    {
        auto user = find_user(id.uid);
        if (!user) return;

        auto it = user->streamers.find(id.collector_id);
        if (it == user->streamers.end()) return;

        it->second->planned = false;
        try_release_streamer(user, it->second);
    }

    void handle_streamers_released(user_ptr user)
    {
        assert(user->streamers.empty());

        if (user->state == user_state::released)
        {
            users_.erase(user->uid);
        }
        else if (user->state == user_state::initial)
        {
            loader_->process(user->shard, { user->uid });
        }
    }

    bool update_acquired_user(const shard_id& shard, const uid& uid)
    {
        auto it = users_.find(uid);
        if (it == users_.end())
        {
            return false;
        }

        auto& user = it->second;
        if (user->shard != shard)
        {
            user->shard = shard;
            if (try_release_user_streamers(user))
            {
                return false;
            }
            user->state = user_state::initial;
            return true;
        }

        if (user->state == user_state::released)
        {
            user->state = user_state::initial;
        }
        else if (user->state == user_state::api_ready)
        {
            user->state = user_state::full_ready;
            run_streamers(user);
        }
        return true;
    }

    void add_streamers(user_ptr user, const collector_info_chunk& chunk)
    {
        for (auto& collector_info : chunk)
        {
            if (user->uid != collector_info.dst_uid)
            {
                YLOG_L(error) << "add_streamers called with inconsistent data";
                continue;
            }

            auto streamer_it = user->streamers.find(collector_info.id);
            if (streamer_it == user->streamers.end())
            {
                YLOG_L(info) << "loaded collector "
                             << meta::get_logging_info(collector_info).to_string();
                auto new_streamer = std::make_shared<streamer_data>(reactor_.io(), collector_info);
                auto res = user->streamers.emplace(collector_info.id, new_streamer);
                streamer_it = res.first;
            }
            else
            {
                YLOG_L(info) << "ignore loaded existing collector "
                             << meta::get_logging_info(collector_info).to_string();
            }
        }

        if (user->state == user_state::full_ready)
        {
            run_streamers(user);
        }
    }

    void run_streamers(user_ptr user)
    {
        for (auto& [id, streamer] : user->streamers)
        {
            auto self = yplatform::shared_from(this);
            streamer->planned = true;
            planner_->add(
                streamer->global_id(),
                io_->wrap([id = streamer->global_id(), this, self](
                              context_ptr ctx, no_data_cb planner_cb) {
                    post_operation(
                        ctx,
                        id,
                        operation::exec,
                        [planner_cb, settings = settings_](
                            context_ptr ctx,
                            streamer_data_ptr streamer,
                            const no_data_cb& queue_cb) {
                            multi_cb cb{ queue_cb, planner_cb };
                            if (!streamer)
                            {
                                return cb(code::streamer_data_not_found);
                            }

                            ctx->logger().append_log_prefix(
                                ctx->uniq_id() + " " +
                                meta::get_logging_info(streamer->collector_info).to_string());
                            operations::spawn<operations::main_op>(ctx, streamer, settings, cb);
                        });
                }));
        }
    }

    // Called from on_release_account/shard, and indirectly from on_acquire_accounts, when user
    // moved to different shard
    bool try_release_user_streamers(user_ptr user)
    {
        if (user->state == user_state::initial)
        {
            loader_->cancel(user->shard, user->uid);
        }

        user->state = user_state::released;
        if (user->streamers.empty())
        {
            handle_streamers_released(user);
            return true;
        }

        std::vector<streamer_data_ptr> streamers;
        for (auto& [id, streamer] : user->streamers)
        {
            streamers.push_back(streamer);
        }

        bool all_released = true;
        for (auto& streamer : streamers)
        {
            all_released = try_release_streamer(user, streamer) && all_released;
        }
        return all_released;
    }

    // Called from try_release_user_streamers to initiate releasing
    // Called from handle_planner_remove - every streamer removed from planner should be released
    // Called from handle_op_finish on released users (after op finished should try release streamer
    // again)
    bool try_release_streamer(user_ptr user, streamer_data_ptr streamer)
    {
        if (streamer->planned)
        {
            planner_->remove(streamer->global_id());
            return false;
        }

        if (streamer->op != operation::noop)
        {
            return false;
        }

        for (auto& op : streamer->operations_queue)
        {
            op(code::user_released);
        }
        user->streamers.erase(streamer->collector_info.id);

        if (user->streamers.empty())
        {
            handle_streamers_released(user);
        }
        return true;
    }

    user_ptr find_user(const uid& uid)
    {
        auto it = users_.find(uid);
        if (it != users_.end())
        {
            return it->second;
        }
        return {};
    }

    void process_streamer_queue(streamer_data_ptr streamer)
    {
        if (streamer->op == operation::exec)
        {
            planner_->stop(streamer->global_id());
        }
        if (streamer->op != operation::noop) return;

        if (streamer->operations_queue.empty())
        {
            return;
        }

        auto op = std::move(streamer->operations_queue.front());
        streamer->operations_queue.pop_front();

        [[maybe_unused]] auto user_it = users_.find(streamer->collector_info.dst_uid);
        assert(user_it != users_.end());
        assert(
            user_it->second->state == user_state::full_ready ||
            (user_it->second->state == user_state::api_ready && op.get_type() != operation::exec));

        if (op.get_type() == operation::exec && streamer->operations_queue.size())
        {
            streamer->operations_queue.emplace_back(std::move(op)); // exec must be executed last
            op = std::move(streamer->operations_queue.front());
            streamer->operations_queue.pop_front();
        }
        streamer->op = op.get_type();
        auto self = yplatform::shared_from(this);
        streamer->io->post([op, streamer, this, self]() mutable {
            op(streamer,
               io_->wrap(std::bind(&streamer_impl::handle_op_finish, self, streamer->global_id())));
        });
    }

    yplatform::reactor& reactor_;
    boost::asio::io_service* io_;
    PlannerPtr planner_;
    LoaderPtr loader_;
    MacsModulePtr macs_;
    streamer_settings_ptr settings_;

    std::map<uid, user_ptr> users_;
    std::map<uid, std::vector<user_cb>> load_handlers_;
};

}
