#pragma once

#include "meta/repository.h"
#include <common/types.h>

namespace collectors::streamer {

using loaded_users = std::map<uid, collector_info_chunk>;
using loaded_users_cb = std::function<void(const shard_id&, const loaded_users&)>;

template <typename LoadOp>
class streamers_loader : public std::enable_shared_from_this<streamers_loader<LoadOp>>
{
    using queue_type = std::map<uid, bool>;

public:
    streamers_loader(boost::asio::io_context* io, uint32_t chunk_size, LoadOp&& load_op)
        : io_(io), chunk_size_(chunk_size), load_op_(load_op)
    {
    }

    void set_handler(const loaded_users_cb& handler)
    {
        auto self = yplatform::shared_from(this);
        io_->post([handler, this, self]() mutable { handler_ = std::move(handler); });
    }

    void process(const shard_id& shard, const uids& uids)
    {
        auto self = yplatform::shared_from(this);
        io_->post([shard, uids, this, self]() mutable {
            for (auto& uid : uids)
            {
                auto it = normal_queues_[shard].find(uid);
                if (it == normal_queues_[shard].end())
                {
                    normal_queues_[shard].insert({ uid, false });
                }
                else
                {
                    it->second = false;
                }
            }
            process_loading_queues();
        });
    }

    void process_prior(const shard_id& shard, const uids& uids)
    {
        auto self = yplatform::shared_from(this);
        io_->post([shard, uids, this, self]() mutable {
            for (auto& uid : uids)
            {
                auto it = priority_queues_[shard].find(uid);
                if (it == priority_queues_[shard].end())
                {
                    priority_queues_[shard].insert({ uid, false });
                }
                else
                {
                    it->second = false;
                }
            }
            process_loading_queues();
        });
    }

    void cancel(const shard_id& shard, const uid& uid)
    {
        auto self = yplatform::shared_from(this);
        io_->post([shard, uid, this, self]() mutable {
            auto normal_it = normal_queues_[shard].find(uid);
            if (normal_it != normal_queues_[shard].end())
            {
                normal_it->second = true;
            }

            auto prior_it = priority_queues_[shard].find(uid);
            if (prior_it != priority_queues_[shard].end())
            {
                prior_it->second = true;
            }
        });
    }

    void cancel(const shard_id& shard)
    {
        auto self = yplatform::shared_from(this);
        io_->post([shard, this, self]() mutable {
            for (auto& [uid, cancelled] : normal_queues_[shard])
            {
                cancelled = true;
            }

            for (auto& [uid, cancelled] : priority_queues_[shard])
            {
                cancelled = true;
            }
        });
    }

private:
    void process_loading_queues()
    {
        for (auto& [shard, queue] : priority_queues_)
        {
            if (!queue.empty() && processing_shards_.count(shard) == 0)
            {
                uids uids;
                uids.reserve(chunk_size_);
                fill_uids_from_queue(uids, queue);
                if (uids.size() < chunk_size_)
                {
                    fill_uids_from_queue(uids, normal_queues_[shard]);
                }

                if (uids.size())
                {
                    start_loading(shard, std::move(uids));
                }
            }
        }

        for (auto& [shard, queue] : normal_queues_)
        {
            if (!queue.empty() && processing_shards_.count(shard) == 0)
            {
                uids uids;
                uids.reserve(chunk_size_);
                fill_uids_from_queue(uids, queue);

                if (uids.size())
                {
                    start_loading(shard, std::move(uids));
                }
            }
        }
    }

    void fill_uids_from_queue(uids& uids, queue_type& queue)
    {
        auto it = queue.begin();
        while (it != queue.end())
        {
            if (uids.size() >= chunk_size_) return;

            auto& [uid, cancelled] = *it;

            if (!cancelled)
            {
                uids.emplace_back(uid);
                it++;
            }
            else
            {
                it = queue.erase(it);
            }
        }
    }

    void start_loading(const shard_id& shard, uids&& uids)
    {
        processing_shards_.insert(shard);
        auto self = yplatform::shared_from(this);
        auto load_cb =
            [shard = shard, uids, this, self](const error& ec, const collector_info_chunk& chunk) {
                processing_shards_.erase(shard);
                if (!ec)
                {
                    loaded_users res;
                    for (auto& uid : uids)
                    {
                        bool cancelled = true;
                        auto normal_it = normal_queues_[shard].find(uid);
                        if (normal_it != normal_queues_[shard].end())
                        {
                            cancelled = normal_it->second;
                            normal_queues_[shard].erase(normal_it);
                        }

                        auto prior_it = priority_queues_[shard].find(uid);
                        if (prior_it != priority_queues_[shard].end())
                        {
                            cancelled &= prior_it->second;
                            priority_queues_[shard].erase(prior_it);
                        }

                        if (!cancelled)
                        {
                            res.insert({ uid, {} });
                        }
                    }
                    for (auto& collector_info : chunk)
                    {
                        auto it = res.find(collector_info.dst_uid);
                        if (it != res.end())
                        {
                            it->second.emplace_back(std::move(collector_info));
                        }
                    }
                    handler_(shard, std::move(res));
                }
                process_loading_queues();
            };
        load_op_(shard, uids, io_->wrap(load_cb));
    }

    boost::asio::io_context* io_;
    uint32_t chunk_size_;
    std::map<shard_id, queue_type> normal_queues_;
    std::map<shard_id, queue_type> priority_queues_;
    std::set<shard_id> processing_shards_;
    loaded_users_cb handler_;
    std::decay_t<LoadOp> load_op_;
};

}
