#pragma once

#include <common/folder.h>
#include <common/label.h>
#include <common/message.h>
#include <common/types.h>

#include <map>
#include <memory>
#include <string>

namespace collectors::streamer {

inline bool compare_cached_folder(const folder& f1, const folder& f2)
{
    if (f1.symbol.size() || f2.symbol.size())
    {
        return f1.symbol < f2.symbol;
    }
    else
    {
        return std::tie(f1.name, f1.parent_fid) < std::tie(f2.name, f2.parent_fid);
    }
}

inline bool compare_cached_label(const label& l1, const label& l2)
{
    if (l1.symbol.size() || l2.symbol.size())
    {
        return l1.symbol < l2.symbol;
    }
    else
    {
        return std::tie(l1.name, l1.type) < std::tie(l2.name, l2.type);
    }
}

struct sync_state
{
    sync_state(const mid& last_mid, const std::vector<std::string>& rpop_ids)
        : last_mid(last_mid), rpop_ids(rpop_ids)
    {
    }

    std::string dst_email;
    std::string src_email;
    std::map<fid, fid> folders_mapping;
    std::map<lid, lid> labels_mapping;

    std::set<folder, decltype(&compare_cached_folder)> cached_folders{ &compare_cached_folder };
    std::set<label, decltype(&compare_cached_label)> cached_labels{ &compare_cached_label };
    std::map<fid, folder> cached_src_folders;
    messages cached_messages;
    mid last_mid;

    message last_message;
    std::uint32_t last_message_retries = 0;

    std::vector<std::string> rpop_ids;
    std::string rpop_info;

    std::uint32_t sent_count = 0;

    void clear_folders()
    {
        cached_folders.clear();
        cached_src_folders.clear();
        folders_mapping.clear();
    }

    void clear_labels()
    {
        cached_labels.clear();
        labels_mapping.clear();
    }

    void clear_messages()
    {
        cached_messages.clear();
    }

    void clear_cache()
    {
        clear_folders();
        clear_labels();
        clear_messages();
    }

    std::string get_src_login()
    {
        return src_email.substr(0, src_email.find("@"));
    }
};

using sync_state_ptr = std::shared_ptr<sync_state>;

}
