#pragma once

#include "settings.h"

#include <common/errors.h>

#include <ymod_httpclient/cluster_client.h>
#include <ymod_httpclient/request.h>
#include <ymod_webserver/request.h>
#include <ymod_webserver/response.h>

#include <yplatform/util/split.h>
#include <yplatform/util/sstream.h>

namespace collectors::web {

inline std::string append_params_to_uri(const std::string& uri, const yhttp::params_list& params)
{
    auto prefix = uri.find('?') != std::string::npos ? '&' : '?';
    return uri + yhttp::url_encode(params, prefix);
}

inline yhttp::request make_proxy_request(
    const std::string& host,
    ymod_webserver::request_ptr req,
    settings_ptr settings,
    const yhttp::params_list& additional_params = {})
{
    auto uri = host + yplatform::util::split(req->raw_request_line, " ").at(1);
    std::string headers;
    yplatform::sstream headers_stream(headers);
    for (auto& [name, value] : req->headers)
    {
        if (!settings->remove_request_headers.count(name))
        {
            headers_stream << name << ": " << value << "\r\n";
        }
    }
    uri = append_params_to_uri(uri, additional_params);

    if (req->method == ymod_webserver::methods::mth_post)
    {
        return yhttp::request::POST(uri, headers, { req->raw_body.begin(), req->raw_body.end() });
    }
    else
    {
        return yhttp::request::GET(uri, headers);
    }
}

template <typename Handler>
inline void forward_request(
    const std::string& cluster_client_name,
    ymod_webserver::request_ptr req,
    settings_ptr settings,
    Handler&& handler,
    const yhttp::params_list& additional_params = {})
{
    auto fwd_request = make_proxy_request("", req, settings, additional_params);
    auto client = yplatform::find<yhttp::cluster_client>(cluster_client_name);
    client->async_run(req->ctx(), fwd_request, std::forward<Handler>(handler));
}

inline void forward_response(
    ymod_webserver::http::stream_ptr stream,
    const yhttp::response& http_resp,
    settings_ptr settings)
{
    stream->set_code(ymod_webserver::codes::code(http_resp.status));
    for (auto& header : settings->proxy_response_headers)
    {
        auto it = http_resp.headers.find(header);
        if (it != http_resp.headers.end())
        {
            stream->add_header(header, it->second);
        }
    }
    stream->ctx()->custom_log_data["status"] = "proxy";
    stream->result_body(http_resp.body);
}

template <typename Handler>
inline void proxy_request(
    const std::string& cluster_client_name,
    ymod_webserver::http::stream_ptr stream,
    settings_ptr settings,
    Handler&& handler,
    const yhttp::params_list& additional_params = {})
{
    forward_request(
        cluster_client_name,
        stream->request(),
        settings,
        [stream, handler = std::forward<Handler>(handler), settings](
            error ec, const yhttp::response& http_resp) {
            if (ec) return handler(ec, stream);

            forward_response(stream, http_resp, settings);
            handler(code::ok, stream);
        },
        additional_params);
}

template <typename Handler>
inline void proxy_rpop_api_request(
    ymod_webserver::http::stream_ptr stream,
    settings_ptr settings,
    Handler handler)
{
    proxy_request(
        "rpop_client",
        stream,
        settings,
        handler,
        { { "_", stream->ctx()->uniq_id() }, { "json", 1 } });
}

}
