#pragma once

#include "settings.h"

#include <common/errors.h>

#include <ymod_httpclient/cluster_client.h>

namespace collectors::web {

namespace {

bool is_error(const json::value& json)
{
    auto error = json["error"];
    return !error.isNull();
}

std::string extract_error_description(const json::value& error)
{
    auto description = error["description"];
    return description.isNull() ? "no description" : description.asString();
}

string_opt extract_application(const json::value& json)
{
    auto token = json["token"];
    if (token.isNull()) return {};
    auto application = token["application"];
    return application.isNull() ? string_opt() : application.asString();
}

string_opt extract_token_value(const json::value& json)
{
    auto token = json["token"];
    if (token.isNull()) return {};
    auto token_value = token["value"];
    return token_value.isNull() ? string_opt() : token_value.asString();
}

error parse_error_from_oauth_response(const std::string& body)
{
    try
    {
        auto json = json::from_string(body);
        auto error_string = json["error"].asString();
        auto description = json["error_description"].asString();
        if (error_string == "invalid_grant")
        {
            if (description == "login or password is not valid")
            {
                return code::invalid_login_or_password;
            }
            return code::invalid_grant;
        }
        else if (error_string == "403")
        {
            if (description == "Expired password")
            {
                return code::expired_password;
            }
            else if (description == "Password change required")
            {
                return code::password_change_required;
            }
        }
    }
    catch (...)
    {
    }
    return code::get_auth_token_error;
}

}

template <typename Handler>
void get_auth_token(
    context_ptr ctx,
    const std::string& login,
    const std::string& password,
    const std::string& ip_addr,
    settings_ptr settings,
    Handler&& handler)
{
    auto oauth_client = yplatform::find<yhttp::cluster_client>("oauth_client");
    auto req = yhttp::request::POST(
        "/token",
        yhttp::form_encode({ { "client_id", settings->oauth_client_id },
                             { "client_secret", settings->oauth_client_secret },
                             { "grant_type", "password" },
                             { "username", login },
                             { "user_ip", ip_addr },
                             { "password", password } }));
    oauth_client->async_run(
        ctx,
        req,
        [ctx, handler = std::forward<Handler>(handler)](
            error ec, const yhttp::response& http_resp) mutable {
            std::string res;
            if (ec)
            {
                TASK_LOG(ctx, error) << "get_auth_token error: " << ec.message();
                return handler(code::get_auth_token_error, res);
            }

            if (http_resp.status / 100 != 2)
            {
                YLOG_CTX_GLOBAL(ctx, error)
                    << "get_auth_token bad http response: " << http_resp.status
                    << ", body: " << http_resp.body;
                ec = parse_error_from_oauth_response(http_resp.body);
            }
            else
            {
                auto json = json::from_string(http_resp.body);
                res = json["access_token"].asString();
            }
            handler(ec, res);
        });
}

template <typename Handler>
void get_auth_token(
    context_ptr ctx,
    const std::string& task_id,
    settings_ptr settings,
    Handler&& handler)
{
    auto social_client = yplatform::find<yhttp::cluster_client>("social_client");
    auto params = yhttp::url_encode({ { "consumer", settings->social_consumer } });
    auto req = yhttp::request::GET("/api/task/" + task_id + params);
    social_client->async_run(
        ctx,
        req,
        [ctx, settings, handler = std::forward<Handler>(handler)](
            error ec, const yhttp::response& http_resp) mutable {
            std::string res;
            if (ec)
            {
                TASK_LOG(ctx, error) << "get_auth_token error: " << ec.message();
                return handler(code::get_auth_token_error, res);
            }

            if (http_resp.status / 100 != 2)
            {
                YLOG_CTX_GLOBAL(ctx, error)
                    << "get_auth_token bad http response: " << http_resp.status
                    << ", body: " << http_resp.body;
                return handler(code::get_auth_token_error, res);
            }

            try
            {
                auto json = json::from_string(http_resp.body);

                if (is_error(json))
                {
                    auto description = extract_error_description(json["error"]);
                    YLOG_CTX_GLOBAL(ctx, error) << "get_auth_token error: " << description;
                    return handler(code::get_auth_token_error, res);
                }

                auto application_opt = extract_application(json);
                if (!application_opt)
                {
                    YLOG_CTX_GLOBAL(ctx, error)
                        << "get_auth_token error: response doesn't contain application: "
                        << http_resp.body;
                    return handler(code::get_auth_token_error, res);
                }

                if (*application_opt != settings->social_application)
                {
                    YLOG_CTX_GLOBAL(ctx, error)
                        << "get_auth_token error: wrong social_task application: "
                        << *application_opt;
                    return handler(code::wrong_social_task, res);
                }

                auto token_value = extract_token_value(json);
                if (!token_value)
                {
                    YLOG_CTX_GLOBAL(ctx, error)
                        << "get_auth_token error: response doesn't contain token: "
                        << http_resp.body;
                    return handler(code::get_auth_token_error, res);
                }
                res = *token_value;
            }
            catch (const std::exception& e)
            {
                YLOG_CTX_GLOBAL(ctx, error)
                    << "get_auth_token error: exception during handling response from social: "
                    << e.what();
                ec = code::get_auth_token_error;
            }
            handler(ec, res);
        });
}

}
