#pragma once

#include "response.h"

#include <common/json.h>
#include <common/types.h>
#include <passport/client.h>
#include <streamer/module.h>
#include <web/get_auth_token.h>

#include <yplatform/coroutine.h>
#include <yplatform/yield.h>

namespace collectors::web::internal {

struct migrate_collector_op
{
    using yield_ctx = yplatform::yield_context<migrate_collector_op>;

    migrate_collector_op(
        settings_ptr settings,
        ymod_webserver::http::stream_ptr http_stream,
        const std::string& /*dst_suid*/, // uid is required, so we don't need suid
        const std::string& dst_uid,
        const std::string& login,
        const std::string& password,
        const std::string& auth_token,
        const mid& last_mid,
        const mids& skipped_mids,
        uint64_t old_popid,
        std::time_t creation_ts,
        bool ignore_folders_struct,
        const std::string& original_server,
        bool enabled,
        const std::string& collector_ip,
        bool ignore_invalid_credentials)
        : settings(settings)
        , http_stream(http_stream)
        , login(login)
        , password(password)
        , collector_ip(collector_ip)
        , ignore_invalid_credentials(ignore_invalid_credentials)
    {
        passport = passport::make_passport_client(http_stream->ctx());
        streamer = yplatform::find<streamer::module>("streamer");
        global_id.uid = dst_uid;
        migrated_collector_draft.dst_uid = dst_uid;
        migrated_collector_draft.auth_token = auth_token;
        migrated_collector_draft.root_folder_id = EMPTY_ID;
        migrated_collector_draft.label_id = EMPTY_ID;
        migrated_collector_draft.last_mid = last_mid;
        migrated_collector_draft.skipped_mids = skipped_mids;
        migrated_collector_draft.old_popid = old_popid;
        migrated_collector_draft.creation_ts = creation_ts;
        migrated_collector_draft.ignore_folders_struct = ignore_folders_struct;
        migrated_collector_draft.original_server = original_server;
        migrated_collector_draft.target_state =
            enabled ? collector_state::created : collector_state::disabled;
    }

    void operator()(yield_ctx ctx)
    {
        reenter(ctx)
        {
            yield passport->get_userinfo_by_login(login, ctx.capture(ec, user_info));
            if (ec == code::user_not_found && ignore_invalid_credentials)
            {
                YLOG_CTX_LOCAL(http_stream->ctx(), info)
                    << "migrate collector for non-existing user: " << login;
                migrated_collector_draft.src_uid = get_non_existing_uid();
            }
            else
            {
                if (ec) yield break;

                migrated_collector_draft.src_uid = user_info.uid;
                if (migrated_collector_draft.auth_token.empty())
                {
                    yield get_auth_token(
                        http_stream->ctx(),
                        login,
                        password,
                        collector_ip,
                        settings,
                        ctx.capture(ec, token));
                    if (ec && !should_ignore_error(ec)) yield break;
                    if (!ec)
                    {
                        migrated_collector_draft.auth_token = token;
                    }
                }

                if (migrated_collector_draft.auth_token.size())
                {
                    yield passport->check_auth_token(
                        migrated_collector_draft.auth_token,
                        { methods::get_real_ip(http_stream), methods::get_real_port(http_stream) },
                        ctx.capture(ec, user_info));
                    while (ec &&
                           blackbox_retries++ <
                               settings->retries_count) // possible read-after-write inconsistency
                    {
                        yield wait(settings->retries_delay, ctx.capture(ec));
                        yield passport->check_auth_token(
                            migrated_collector_draft.auth_token,
                            { methods::get_real_ip(http_stream),
                              methods::get_real_port(http_stream) },
                            ctx.capture(ec, user_info));
                    }

                    if (migrated_collector_draft.src_uid != user_info.uid)
                    {
                        YLOG_CTX_LOCAL(http_stream->ctx(), error)
                            << "auth_token for different user: src_uid="
                            << migrated_collector_draft.src_uid << ", token_uid=" << user_info.uid;
                        ec = code::invalid_auth_token;
                        yield break;
                    }
                }
            }

            yield streamer->migrate_collector(
                http_stream->ctx(), migrated_collector_draft, ctx.capture(ec, id));
            if (ec)
            {
                yield break;
            }
            else
            {
                global_id = global_collector_id(migrated_collector_draft.dst_uid, id);
            }

            respond(http_stream, global_id.to_string());
        }
        if (ctx.is_complete()) complete();
    }

    void operator()(yield_ctx::exception_type exception)
    {
        ec = make_error(exception);
        YLOG_CTX_LOCAL(http_stream->ctx(), error)
            << "exception during migrate_collector_op: " << error_message(ec);
        complete();
    }

    void complete()
    {
        typed::log_api(http_stream->ctx(), global_id, "migrate_collector", ec);
        if (ec)
        {
            YLOG_CTX_LOCAL(http_stream->ctx(), error)
                << "migrate_collector error: " << ec.message();
            respond(http_stream, ec);
        }
    }

    bool should_ignore_error(error ec)
    {
        auto invalid_credentials_error =
            (ec == code::password_change_required) || (ec == code::invalid_login_or_password);
        return invalid_credentials_error && ignore_invalid_credentials;
    }

    template <typename Handler>
    void wait(const time_traits::duration& duration, Handler&& h)
    {
        auto timer = http_stream->make_timer();
        timer->expires_from_now(duration);
        timer->async_wait([timer, h = std::move(h)](auto ec) mutable {
            if (ec != boost::asio::error::operation_aborted)
            {
                h(ec);
            }
        });
    }

    std::string get_non_existing_uid() const
    {
        static const std::string uid = std::to_string(std::numeric_limits<int64_t>::max());
        return uid;
    }

    settings_ptr settings;
    ymod_webserver::http::stream_ptr http_stream;
    passport::client_ptr passport;
    streamer::module_ptr streamer;

    migrated_collector_draft migrated_collector_draft;
    std::string login;
    std::string password;
    std::string collector_ip;
    bool ignore_invalid_credentials;

    user_info user_info;
    collector_id id;
    global_collector_id global_id;

    int blackbox_retries = 0;
    std::string token;
    error ec;
};

}
