#pragma once

#include "util.h"

#include <passport/client.h>
#include <streamer/module.h>
#include <web/settings.h>
#include <web/get_auth_token.h>

#include <yplatform/coroutine.h>
#include <yplatform/yield.h>

namespace collectors::web::methods {

struct create_yandex_collector_op
{
    using yield_ctx = yplatform::yield_context<create_yandex_collector_op>;

    create_yandex_collector_op(
        settings_ptr settings,
        ymod_webserver::http::stream_ptr http_stream,
        const std::string& /*dst_suid*/,
        const std::string& dst_uid,
        const std::string& social_task_id,
        const fid& root_folder_id,
        const lid& label_id)
        : settings(settings)
        , http_stream(http_stream)
        , dst_uid(dst_uid)
        , social_task_id(social_task_id)
    {
        passport = passport::make_passport_client(http_stream->ctx());
        streamer = yplatform::find<streamer::module>("streamer");
        global_id.uid = dst_uid;
        new_collector_draft.dst_uid = dst_uid;
        new_collector_draft.root_folder_id = root_folder_id;
        new_collector_draft.label_id = label_id;
    }

    void operator()(yield_ctx ctx)
    {
        reenter(ctx)
        {
            yield get_auth_token(
                http_stream->ctx(), social_task_id, settings, ctx.capture(ec, token));
            if (ec)
            {
                error_context = "get auth token error";
                yield break;
            }
            new_collector_draft.auth_token = token;
            yield passport->check_auth_token(
                new_collector_draft.auth_token,
                { get_real_ip(http_stream), get_real_port(http_stream) },
                ctx.capture(ec, user_info));
            if (ec)
            {
                error_context = "check auth token error";
                yield break;
            }

            new_collector_draft.src_uid = user_info.uid;
            yield streamer->create_collector(
                http_stream->ctx(), new_collector_draft, ctx.capture(ec, id));
            if (ec)
            {
                error_context = "create collector error";
                yield break;
            }

            global_id = global_collector_id(dst_uid, id);
            {
                json::value resp;
                resp["popid"] = global_id.to_string();
                resp["email"] = user_info.email;
                respond(http_stream, std::move(resp));
            }
        }
        if (ctx.is_complete()) complete();
    }

    void operator()(yield_ctx::exception_type exception)
    {
        ec = make_error(exception);
        YLOG_CTX_LOCAL(http_stream->ctx(), error)
            << "exception during create_yandex_collector_op: " << error_message(ec);
        complete();
    }

    void complete()
    {
        typed::log_api(http_stream->ctx(), global_id, "create_yandex_collector", ec);
        if (ec) respond(http_stream, ec, error_context);
    }

    settings_ptr settings;
    ymod_webserver::http::stream_ptr http_stream;
    passport::client_ptr passport;
    streamer::module_ptr streamer;

    uid dst_uid;
    std::string social_task_id;
    new_collector_draft new_collector_draft;

    error ec;
    std::string error_context;
    user_info user_info;
    std::string token;
    collector_id id;
    global_collector_id global_id;
};

}
