#pragma once

#include <common/json.h>
#include <common/types.h>
#include <streamer/module.h>
#include <web/forward_request.h>

#include <ymod_webserver/response.h>
#include <yplatform/coroutine.h>
#include <yplatform/yield.h>

namespace collectors::web::methods {

struct delete_collector_op
{
    using yield_ctx = yplatform::yield_context<delete_collector_op>;

    delete_collector_op(
        settings_ptr settings,
        ymod_webserver::http::stream_ptr http_stream,
        const std::string& suid,
        const std::string& /*uid*/,
        const std::string& popid)
        : settings(settings), http_stream(http_stream), suid(suid), id(popid)
    {
        streamer = yplatform::find<streamer::module>("streamer");
    }

    void operator()(yield_ctx ctx)
    {
        reenter(ctx)
        {
            yield streamer->get_collector_info(
                http_stream->ctx(), id, ctx.capture(ec, collector_info));
            if (ec)
            {
                error_context = "get collector info error";
                yield break;
            }

            if (collector_info.old_popid)
            {
                yield delete_old_collector(collector_info.old_popid, ctx.capture(ec, http_resp));
                if (ec)
                {
                    error_context = "delete old collector error";
                    yield break;
                }

                result = json::from_string(http_resp.body);
                if (result.isMember("error"))
                {
                    if (result["error"].isMember("description"))
                    {
                        YLOG_CTX_LOCAL(http_stream->ctx(), error)
                            << "rpop error: " << result["error"]["description"].asString();
                    }
                    else
                    {
                        YLOG_CTX_LOCAL(http_stream->ctx(), error)
                            << "rpop bad response: " << http_resp.body;
                    }
                    forward_response(http_stream, http_resp, settings);
                    yield break;
                }
            }

            yield streamer->delete_collector(http_stream->ctx(), id, ctx.capture(ec));
            if (ec)
            {
                error_context = "delete collector error";
                yield break;
            }

            respond(http_stream);
        }
        if (ctx.is_complete()) complete();
    }

    void operator()(yield_ctx::exception_type exception)
    {
        ec = make_error(exception);
        YLOG_CTX_LOCAL(http_stream->ctx(), error)
            << "exception during delete_collector_op: " << error_message(ec);
        complete();
    }

    void complete()
    {
        typed::log_api(http_stream->ctx(), id, "delete_collector", ec);
        if (ec) respond(http_stream, ec, error_context);
    }

    template <typename Handler>
    void delete_old_collector(uint64_t popid, Handler&& handler)
    {
        auto client = yplatform::find<yhttp::cluster_client>("rpop_client");
        auto req = yhttp::request::GET(
            "/api/delete" +
            yhttp::url_encode(
                { { "json", 1 }, { "mdb", "pg" }, { "suid", suid }, { "popid", popid } }));
        client->async_run(http_stream->ctx(), req, std::forward<Handler>(handler));
    }

    settings_ptr settings;
    ymod_webserver::http::stream_ptr http_stream;
    streamer::module_ptr streamer;

    std::string error_context;
    std::string suid;
    global_collector_id id;
    collector_info collector_info;
    json::value result;
    yhttp::response http_resp;
    error ec;
};

}
