#pragma once

#include "util.h"

#include <common/types.h>
#include <streamer/module.h>
#include <web/get_auth_token.h>
#include <web/settings.h>

#include <yplatform/coroutine.h>
#include <yplatform/yield.h>

namespace collectors::web::methods {

struct edit_collector_op
{
    using yield_ctx = yplatform::yield_context<edit_collector_op>;

    edit_collector_op(
        settings_ptr settings,
        ymod_webserver::http::stream_ptr http_stream,
        const std::string& /*suid*/,
        const std::string& /*uid*/,
        const std::string& popid,
        const std::optional<std::string>& social_task_id,
        const std::optional<std::string>& login,
        const std::optional<std::string>& password,
        const std::optional<fid>& root_folder_id,
        const std::optional<lid>& label_id)
        : settings(settings)
        , http_stream(http_stream)
        , id(popid)
        , social_task_id(social_task_id)
        , login(login)
        , password(password)
        , root_folder_id(root_folder_id)
        , label_id(label_id)
    {
        passport = passport::make_passport_client(http_stream->ctx());
        streamer = yplatform::find<streamer::module>("streamer");
    }

    void operator()(yield_ctx ctx)
    {
        reenter(ctx)
        {
            if (social_task_id)
            {
                yield get_auth_token(
                    http_stream->ctx(), *social_task_id, settings, ctx.capture(ec, auth_token));
                if (ec)
                {
                    error_context = "get auth token error";
                    yield break;
                }

                yield passport->check_auth_token(
                    *auth_token,
                    { get_real_ip(http_stream), get_real_port(http_stream) },
                    ctx.capture(ec, user_info));
                if (ec)
                {
                    error_context = "check auth token error";
                    yield break;
                }

                yield streamer->get_collector_info(
                    http_stream->ctx(), id, ctx.capture(ec, collector_info));
                if (ec)
                {
                    error_context = "get collector info error";
                    yield break;
                }

                if (collector_info.src_uid != user_info.uid)
                {
                    ec = code::wrong_social_task;
                    yield break;
                }
            }
            else if (login && password)
            {
                yield passport->get_userinfo_by_login(*login, ctx.capture(ec, user_info));
                if (ec)
                {
                    error_context = "get userinfo by login error";
                    yield break;
                }

                yield streamer->get_collector_info(
                    http_stream->ctx(), id, ctx.capture(ec, collector_info));
                if (ec)
                {
                    error_context = "get collector info error";
                    yield break;
                }

                if (collector_info.src_uid != user_info.uid)
                {
                    ec = code::wrong_login;
                    yield break;
                }

                yield get_auth_token(
                    http_stream->ctx(),
                    *login,
                    *password,
                    get_real_ip(http_stream),
                    settings,
                    ctx.capture(ec, token));
                if (ec)
                {
                    error_context = "get auth token error";
                    yield break;
                }

                auth_token = token;
            }

            yield streamer->edit_collector(
                http_stream->ctx(), id, auth_token, root_folder_id, label_id, ctx.capture(ec));
            if (ec)
            {
                error_context = "edit collector error";
            }
        }
        if (ctx.is_complete()) complete();
    }

    void operator()(yield_ctx::exception_type exception)
    {
        ec = make_error(exception);
        YLOG_CTX_LOCAL(http_stream->ctx(), error)
            << "exception during edit_collector_op: " << error_message(ec);
        complete();
    }

    void complete()
    {
        typed::log_api(
            http_stream->ctx(),
            id,
            "edit_collector",
            ec,
            (auth_token ? "with new token" : "no new token"));
        respond(http_stream, ec, error_context);
    }

    settings_ptr settings;
    ymod_webserver::http::stream_ptr http_stream;
    passport::client_ptr passport;
    streamer::module_ptr streamer;

    global_collector_id id;
    std::optional<std::string> social_task_id;
    std::optional<std::string> login;
    std::optional<std::string> password;
    std::optional<fid> root_folder_id;
    std::optional<lid> label_id;

    error ec;
    std::string error_context;
    user_info user_info;
    collector_info collector_info;
    std::optional<std::string> auth_token;
    std::string token;
};

}
