#pragma once

#include "util.h"

#include <common/json.h>
#include <common/types.h>

#include <passport/client.h>
#include <streamer/module.h>
#include <web/settings.h>
#include <web/forward_request.h>

#include <ymod_blackbox/client.h>
#include <ymod_webserver/response.h>

#include <yplatform/coroutine.h>
#include <yplatform/yield.h>

namespace collectors::web::methods {

struct list_collectors_op
{
    using yield_ctx = yplatform::yield_context<list_collectors_op>;

    list_collectors_op(
        settings_ptr settings,
        ymod_webserver::http::stream_ptr http_stream,
        const std::string& owner_suid,
        const std::string& owner_uid,
        const std::string& popid)
        : settings(settings)
        , http_stream(http_stream)
        , owner_suid(owner_suid)
        , owner_uid(owner_uid)
        , popid(popid)
    {
        passport = passport::make_passport_client(http_stream->ctx());
        streamer = yplatform::find<streamer::module>("streamer");
    }

    void operator()(yield_ctx ctx)
    {
        reenter(ctx)
        {
            if (popid.empty() || is_old_popid(popid))
            {
                yield forward_request(
                    "rpop_client",
                    http_stream->request(),
                    settings,
                    ctx.capture(ec, rpop_resp),
                    { { "_", http_stream->ctx()->uniq_id() }, { "json", 1 } });
                if (ec)
                {
                    error_context = "forward request error";
                    yield break;
                }

                result = json::from_string(rpop_resp.body);
                if (result.isMember("error"))
                {
                    if (result["error"].isMember("description"))
                    {
                        YLOG_CTX_LOCAL(http_stream->ctx(), error)
                            << "rpop error: " << result["error"]["description"].asString();
                    }
                    else
                    {
                        YLOG_CTX_LOCAL(http_stream->ctx(), error)
                            << "rpop bad response: " << rpop_resp.body;
                    }
                    forward_response(http_stream, rpop_resp, settings);
                    yield break;
                }
            }

            if (owner_uid.empty())
            {
                yield passport->get_userinfo_by_suid(owner_suid, ctx.capture(ec, passport_info));
                if (ec)
                {
                    error_context = "get userinfo by suid error";
                    yield break;
                }

                owner_uid = passport_info.uid;
            }

            yield streamer->list_collectors(
                http_stream->ctx(), owner_uid, ctx.capture(ec, collector_ids));
            if (ec)
            {
                error_context = "list collectors error";
                yield break;
            }

            while (collector_ids.size())
            {
                current_id = global_collector_id(owner_uid, collector_ids.back());
                current_email = "";
                if (popid.size() && current_id.to_string() != popid)
                {
                    collector_ids.pop_back();
                    continue;
                }

                yield streamer->get_collector_info(
                    http_stream->ctx(), current_id, ctx.capture(ec, current_collector));
                if (ec == code::streamer_data_not_found)
                {
                    ec = code::ok;
                    collector_ids.pop_back();
                    continue;
                }
                if (ec)
                {
                    error_context = "get collector info error";
                    yield break;
                }

                if (current_collector.old_popid)
                {
                    for (std::size_t i = 0; i < result["rpops"].size(); ++i)
                    {
                        auto index = static_cast<int>(i);
                        auto strPopid = result["rpops"][index]["popid"].asString();
                        if (strPopid == std::to_string(current_collector.old_popid))
                        {
                            json::value removed;
                            result["rpops"].removeIndex(index, &removed);
                            current_email = removed["email"].asString();
                            break;
                        }
                    }
                }

                yield passport->get_userinfo_by_uid(
                    current_collector.src_uid, ctx.capture(ec, passport_info));
                if (!ec)
                {
                    current_email = passport_info.email;
                }
                else if (ec == code::user_not_found || ec == code::empty_email)
                {
                    current_email = settings->deleted_email_placeholder;
                    ec = {};
                }
                else if (current_email.size())
                {
                    ec = {};
                }
                if (ec)
                {
                    error_context = "get userinfo by uid error";
                    yield break;
                }

                result["rpops"].append(make_json(current_collector, current_id, current_email));
                collector_ids.pop_back();
            }

            complete();
        }
        if (ctx.is_complete() && ec) complete(); // double call if exception from respond
    }

    void operator()(yield_ctx::exception_type exception)
    {
        ec = make_error(exception);
        YLOG_CTX_LOCAL(http_stream->ctx(), error)
            << "exception during list_collectors_op: " << error_message(ec);
        complete();
    }

    void complete()
    {
        if (ec)
        {
            respond(http_stream, ec, error_context);
        }
        else
        {
            respond(http_stream, result);
        }
    }

    json::value make_json(
        const collector_info& info,
        const global_collector_id& id,
        const std::string& email)
    {
        json::value res;
        res["popid"] = id.to_string();
        // We need to respond original server, to keep compatibility with rpop-info header
        res["server"] = info.original_server;
        // Respond port number according to protocol used in old collectors
        res["port"] = info.ignore_folders_struct ? "995" : "993";
        res["use_ssl"] = true;
        res["last_connect"] = std::to_string(info.last_run_ts);
        res["session_duration"] = "0";
        res["bad_retries"] = 0;
        res["last_msg_count"] = 0;
        res["leave_msgs"] = true;
        res["abook_sync_state"] = "0";
        res["imap"] = !info.ignore_folders_struct;
        res["is_oauth"] = true;
        res["oauth_app"] = "yandex-mail-collector";
        res["mark_archive_read"] = 0;

        res["is_on"] = get_is_on_value(info);
        res["error_status"] = info.auth_token.size() ? "ok" : "login error";
        res["root_folder_id"] = info.root_folder_id;
        res["label_id"] = info.label_id;
        res["email"] = email;
        res["login"] = email;

        return res;
    }

    std::string get_is_on_value(const collector_info& info)
    {
        if (info.auth_token.empty())
        {
            return "3";
        }
        auto state = info.state;
        if (state == collector_state::migrated)
        {
            state = info.migration_target_state;
        }
        return state != collector_state::disabled ? "1" : "0";
    }

    settings_ptr settings;
    ymod_webserver::http::stream_ptr http_stream;
    passport::client_ptr passport;
    streamer::module_ptr streamer;

    std::string error_context;
    yhttp::response rpop_resp;
    std::string owner_suid;
    uid owner_uid;
    std::string popid;
    collector_ids collector_ids;
    global_collector_id current_id;
    std::string current_email;
    collector_info current_collector;

    json::value result;
    user_info passport_info;
    collector_info info;
    error ec;
};

}
