#pragma once

#include <common/types.h>
#include <passport/client.h>
#include <web/forward_request.h>

#include <ymod_httpclient/call.h>
#include <ymod_mdb_sharder/users_distributor.h>
#include <yplatform/util/safe_call.h>

namespace collectors::web::methods {

struct proxy_to_owner_op
{
    using yield_ctx = yplatform::yield_context<proxy_to_owner_op>;

    template <typename Method, typename... Args>
    proxy_to_owner_op(
        Method&& method,
        settings_ptr settings,
        ymod_webserver::http::stream_ptr http_stream,
        const std::string& suid,
        const std::string& uid,
        Args&&... args)
        : method(std::bind(method, http_stream, suid, uid, std::forward<Args>(args)...))
        , settings(settings)
        , http_stream(http_stream)
        , suid(suid)
        , uid(uid)
    {
        passport = passport::make_passport_client(http_stream->ctx());
        users_distributor =
            yplatform::find<ymod_mdb_sharder::users_distributor>("users_distributor");
    }

    void operator()(yield_ctx ctx)
    {
        reenter(ctx)
        {
            if (uid.empty())
            {
                yield passport->get_userinfo_by_suid(suid, ctx.capture(ec, user_info));
                if (ec)
                {
                    error_context = "get userinfo by suid error";
                    yield break;
                }
                else
                {
                    uid = user_info.uid;
                }
            }

            yield users_distributor->get_owner(
                http_stream->ctx(), std::stoull(uid), ctx.capture(ec, node));
            if (ec)
            {
                error_context = "get owner error";
                yield break;
            }

            if (node.id == users_distributor->my_node_id())
            {
                yplatform::safe_call(http_stream->ctx(), method);
            }
            else
            {
                yield yplatform::find<yhttp::call>("http_client")
                    ->async_run(
                        http_stream->ctx(),
                        make_proxy_request(
                            get_proxy_host(),
                            http_stream->request(),
                            settings,
                            { { "_", http_stream->ctx()->uniq_id() } }),
                        ctx.capture(ec, http_resp));
                if (ec)
                {
                    error_context = "http_client error";
                    yield break;
                }

                yplatform::safe_call(
                    http_stream->ctx(), &forward_response, http_stream, http_resp, settings);
            }
        }
        if (ctx.is_complete() && ec) complete();
    }

    void operator()(yield_ctx::exception_type exception)
    {
        ec = make_error(exception);
        YLOG_CTX_LOCAL(http_stream->ctx(), error)
            << "exception during proxy_to_owner_op: " << error_message(ec);
        complete();
    }

    void complete()
    {
        respond(http_stream, ec, error_context);
    }

    std::string get_proxy_host()
    {
        std::string proto = http_stream->is_secure() ? "https" : "http";
        std::string proxy_host =
            proto + "://" + node.host + ":" + std::to_string(http_stream->ctx()->local_port);
        return proxy_host;
    }

private:
    std::function<void(void)> method;
    settings_ptr settings;
    ymod_webserver::http::stream_ptr http_stream;
    std::string suid;
    uid uid;

    passport::client_ptr passport;
    boost::shared_ptr<ymod_mdb_sharder::users_distributor> users_distributor;

    std::string error_context;
    yhttp::response http_resp;
    user_info user_info;
    ymod_mdb_sharder::node_info node;
    error ec;
};

}
