#pragma once

#include <common/json.h>
#include <common/server_info.h>
#include <common/types.h>

#include <ymod_webserver/response.h>

namespace collectors::web::methods {

inline void add_base_response_fields(context_ptr ctx, const std::string& method, json::value& json)
{
    json["request_id"] = ctx->uniq_id();
    json["method"] = method;
    json["host"] = my_host_name;
}

inline json::value make_error_response(context_ptr ctx, const std::string& method, error ec)
{
    json::value error;
    error["description"] = ec.message();
    error["reason"] = ec.message();
    add_base_response_fields(ctx, method, error);

    json::value res;
    res["error"] = error;
    return res;
}

inline void respond(ymod_webserver::http::stream_ptr stream, json::value data = {})
{
    add_base_response_fields(stream->ctx(), stream->request()->url.make_full_path(), data);
    stream->ctx()->custom_log_data["status"] = "ok";
    stream->result(ymod_webserver::codes::ok, json::to_string(data));
}

inline void respond(
    ymod_webserver::http::stream_ptr stream,
    error ec,
    const std::string error_context = "")
{
    if (!ec)
    {
        return respond(stream);
    }

    if (error_context.size())
    {
        LERR_(stream->ctx()) << error_context << ": " << ec.message();
    }

    auto error_response =
        make_error_response(stream->ctx(), stream->request()->url.make_full_path(), ec);
    stream->ctx()->custom_log_data["status"] = "error";
    stream->ctx()->custom_log_data["error_reason"] = ec.message();
    stream->result(
        ymod_webserver::codes::ok, // old API always returns 200
        json::to_string(error_response));
}

inline std::string get_real_ip(ymod_webserver::http::stream_ptr stream)
{
    auto& headers = stream->request()->headers;
    if (auto it = headers.find("x-real-ip"); it != headers.end())
    {
        return it->second;
    }
    else
    {
        return stream->remote_addr().to_string();
    }
}

inline unsigned get_real_port(ymod_webserver::http::stream_ptr stream)
{
    auto req = stream->request();
    auto& headers = req->headers;
    if (auto it = headers.find("x-real-port"); it != headers.end())
    {
        return std::stoi(it->second);
    }
    else
    {
        return req->ctx()->remote_port;
    }
}

}
