#pragma once

#include "binded_spawn.h"
#include "forward_request.h"
#include "methods/create_yandex_collector.h"
#include "methods/delete_collector.h"
#include "methods/edit_collector.h"
#include "methods/update_collector_state.h"
#include "methods/list_collectors.h"
#include "methods/ping.h"
#include "methods/proxy_if_old_op.h"
#include "methods/proxy_to_owner_op.h"
#include "methods/util.h"

#include "internal/check_tvm.h"
#include "internal/folders.h"
#include "internal/labels.h"
#include "internal/next_message_chunk.h"
#include "internal/acquired_shards.h"
#include "internal/acquired_buckets_info.h"
#include "internal/loaded_users.h"
#include "internal/unload_user.h"
#include "internal/migrate_collector.h"
#include "internal/planner.h"
#include "internal/unmigrate_collector.h"

#include "settings.h"

#include <common/errors.h>
#include <common/types.h>

#include <ymod_tvm/settings.h>
#include <ymod_webserver/server.h>
#include <yplatform/module.h>
#include <yplatform/find.h>

#include <boost/optional.hpp>
#include <optional>

namespace collectors::web {

namespace {

template <typename Container>
struct array_converter
{
    Container operator()(const std::string& value) const
    {
        auto splitted = yplatform::util::split(value, ",");
        if constexpr (std::is_same_v<Container, decltype(splitted)>)
        {
            return splitted;
        }
        Container res(splitted.size());

        std::transform(splitted.begin(), splitted.end(), res.begin(), [](auto& entry) {
            return boost::lexical_cast<typename Container::value_type>(entry);
        });
        return res;
    }
};

bool is_num_array_member(char c)
{
    return ::isdigit(c) || c == ',';
};

}

class web : public yplatform::module
{
    using stream_ptr = ymod_webserver::http::stream_ptr;

public:
    void init(const yplatform::ptree& config)
    {
        yplatform::read_ptree(
            settings_->proxy_response_headers, config.get_child("proxy"), "proxy_response_headers");
        yplatform::read_ptree(
            settings_->remove_request_headers, config.get_child("proxy"), "remove_request_headers");
        settings_->oauth_client_id = config.get<std::string>("oauth_client_id");
        settings_->oauth_client_secret = config.get<std::string>("oauth_client_secret");
        settings_->social_application =
            config.get<std::string>("social_application", "yandex-mail-collector");
        settings_->social_consumer = config.get<std::string>("social_consumer");
        settings_->mailbox_settings.internal_api_port = config.get("internal_api.port", 5048);
        settings_->mailbox_settings.internal_api_service_name =
            config.get<std::string>("internal_api.service_name");

        settings_->tvm_module = config.get("tvm_module", "");
        settings_->protect_localhost = config.get("protect_localhost", false);
        settings_->retries_count = config.get("retries_count", settings_->retries_count);
        settings_->retries_delay = config.get("retries_delay", settings_->retries_delay);
        settings_->deleted_email_placeholder =
            config.get("deleted_email_placeholder", settings_->deleted_email_placeholder);

        auto mappings = ymod_tvm::tvm2::settings::read_mappings(config);
        auto range = config.equal_range("allowed_sources");
        for (auto it = range.first; it != range.second; ++it)
        {
            if (auto alias = it->second.get_value(std::string{}); mappings.count(alias))
            {
                settings_->allowed_sources[mappings[alias].id] = alias;
            }
            else
            {
                settings_->allowed_sources[it->second.get<uint32_t>("id")] =
                    it->second.get<std::string>("name");
            }
        }

        using ymod_webserver::transformer;
        using ymod_webserver::argument;
        using ymod_webserver::optional_argument;

        auto proxy_request_handler = [](error ec, stream_ptr stream) {
            if (ec)
            {
                methods::respond(stream, ec, "proxy to rpop error");
            }
        };

        auto proxy_request_method = std::bind(
            proxy_rpop_api_request<decltype(proxy_request_handler)>,
            std::placeholders::_1,
            settings_,
            proxy_request_handler);

        auto webserver = yplatform::find<ymod_webserver::server>("web_server");
        webserver->bind("", { "/ping" }, methods::ping);
        webserver->bind(
            "",
            { "/api/folders",
              "/api/info",
              "/api/status",
              "/api/v2/check_login",
              "/api/v2/check_oauth",
              "/api/v2/setup_login",
              "/api/v2/setup_login",
              "/api/v2/smtp_data" },
            proxy_request_method);

        webserver->bind(
            "",
            { "/api/run", "/api/abook_enable" },
            methods::proxy_if_old_op(
                [](stream_ptr stream,
                   const std::string& /*suid*/,
                   const std::string& /*uid*/,
                   const std::string& /*popid*/) { methods::respond(stream, code::not_supported); },
                settings_),
            transformer(
                optional_argument<std::string>("suid", ""),
                optional_argument<std::string>("uid", ""),
                argument<std::string>("popid")));

        webserver->bind(
            "",
            { "/api/check_server" },
            [proxy_request_method](
                ymod_webserver::http::stream_ptr stream, const std::string& popid) {
                if (popid.size() && !is_old_popid(popid))
                {
                    methods::respond(stream, code::ok);
                }
                else
                {
                    proxy_request_method(stream);
                }
            },
            transformer(optional_argument<std::string>("popid", "")));

        webserver->bind("", { "/api/create" }, proxy_request_method);

        webserver->bind(
            "",
            { "/api/create_yandex" },
            binded_spawn<methods::proxy_to_owner_op>(
                binded_spawn<methods::create_yandex_collector_op>(settings_), settings_),
            transformer(
                optional_argument<std::string>("suid", ""),
                argument<std::string>("uid"),
                argument<std::string>("social_task_id"),
                optional_argument<std::string>("root_folder_id", EMPTY_ID),
                optional_argument<std::string>("label_id", EMPTY_ID)));

        webserver->bind(
            "",
            { "/api/list" },
            binded_spawn<methods::proxy_to_owner_op>(
                binded_spawn<methods::list_collectors_op>(settings_), settings_),
            transformer(
                argument<std::string>("suid"),
                optional_argument<std::string>("uid", ""),
                optional_argument<std::string>("popid", "")));

        webserver->bind(
            "",
            { "/api/delete" },
            methods::proxy_if_old_op(
                binded_spawn<methods::proxy_to_owner_op>(
                    binded_spawn<methods::delete_collector_op>(settings_), settings_),
                settings_),
            transformer(
                argument<std::string>("suid"),
                optional_argument<std::string>("uid", ""),
                argument<std::string>("popid")));

        webserver->bind(
            "",
            { "/api/enable" },
            methods::proxy_if_old_op(
                binded_spawn<methods::proxy_to_owner_op>(
                    &methods::update_collector_state, settings_),
                settings_),
            transformer(
                argument<std::string>("suid"),
                optional_argument<std::string>("uid", ""),
                argument<std::string>("popid"),
                argument<bool>("is_on")));

        webserver->bind(
            "",
            { "/api/edit" },
            methods::proxy_if_old_op(
                binded_spawn<methods::proxy_to_owner_op>(
                    binded_spawn<methods::edit_collector_op>(settings_), settings_),
                settings_),
            transformer(
                argument<std::string>("suid"),
                optional_argument<std::string>("uid", ""),
                argument<std::string>("popid"),
                optional_argument("social_task_id", std::optional<std::string>()),
                optional_argument("login", std::optional<std::string>()),
                optional_argument("password", std::optional<std::string>()),
                optional_argument("root_folder_id", std::optional<std::string>()),
                optional_argument("label_id", std::optional<std::string>())));

        webserver->bind(
            "service",
            { "/migrate" },
            binded_spawn<methods::proxy_to_owner_op>(
                binded_spawn<internal::migrate_collector_op>(settings_), settings_),
            transformer(
                optional_argument<std::string>("dst_suid", ""),
                argument<std::string>("dst_uid"),
                argument<std::string>("login"),
                optional_argument<std::string>("password", ""),
                optional_argument<std::string>("auth_token", ""),
                argument<mid>("last_mid"),
                optional_argument<mids>(
                    "skipped_mids",
                    mids(),
                    ymod_webserver::validator(is_num_array_member),
                    array_converter<mids>()),
                argument<uint64_t>("old_popid"),
                argument<std::time_t>("creation_ts"),
                argument<bool>("ignore_folders_struct"),
                argument<std::string>("original_server"),
                argument<bool>("enabled"),
                argument<std::string>("collector_ip"),
                optional_argument<bool>("ignore_invalid_credentials", false)));
        webserver->bind(
            "service",
            { "/unmigrate" },
            binded_spawn<methods::proxy_to_owner_op>(&internal::unmigrate_collector, settings_),
            transformer(
                optional_argument<std::string>("suid", ""),
                argument<std::string>("uid"),
                argument<std::string>("popid")));

        webserver->bind("service", { "/acquired_shards" }, &internal::get_acquired_shards);
        webserver->bind(
            "service", { "/acquired_buckets_info" }, &internal::get_acquired_buckets_info);
        webserver->bind("service", { "/loaded_users" }, &internal::get_loaded_users);
        webserver->bind(
            "service",
            { "/unload_user" },
            &internal::unload_user,
            transformer(argument<uid>("uid")));
        webserver->bind("service", { "/disable_planner" }, &internal::disable_planner);
        webserver->bind("service", { "/enable_planner" }, &internal::enable_planner);

        webserver->bind(
            "internal",
            { "/folders" },
            internal::check_tvm(
                std::bind(&internal::get_folders, settings_, ph::_1, ph::_2), settings_),
            transformer(argument<uid>("uid")));
        webserver->bind(
            "internal",
            { "/pop3_folders" },
            internal::check_tvm(
                std::bind(&internal::get_pop3_folders, settings_, ph::_1, ph::_2), settings_),
            transformer(argument<uid>("uid")));
        webserver->bind(
            "internal",
            { "/labels" },
            internal::check_tvm(
                std::bind(&internal::get_labels, settings_, ph::_1, ph::_2), settings_),
            transformer(argument<uid>("uid")));
        webserver->bind(
            "internal",
            { "/next_message_chunk" },
            internal::check_tvm(
                std::bind(
                    &internal::get_next_message_chunk, settings_, ph::_1, ph::_2, ph::_3, ph::_4),
                settings_),
            transformer(argument<uid>("uid"), argument<mid>("mid"), argument<uint64_t>("count")));
        webserver->bind(
            "internal",
            { "/pop3_next_message_chunk" },
            internal::check_tvm(
                std::bind(
                    &internal::get_pop3_next_message_chunk,
                    settings_,
                    ph::_1,
                    ph::_2,
                    ph::_3,
                    ph::_4),
                settings_),
            transformer(argument<uid>("uid"), argument<mid>("mid"), argument<uint64_t>("count")));
    }

private:
    settings_ptr settings_ = std::make_shared<settings>();
};

}
