import os.path
import urllib

import requests
import retrying

import json
from library.python import resource
from mail.devpack.lib import helpers
from mail.devpack.lib.components.base import YplatformComponent
from mail.devpack.lib.state import read_state
from mail.devpack.lib.yhttp_service import YplatformHttpService
from mail.collie.devpack.components.colliedb import CollieDb, CollieDbForDirectorySync, CollieDbForSync
from mail.devpack.lib.components.sharpei import Sharpei
from mail.devpack.lib.components.tvmapi import TvmApi


class Collie(YplatformComponent):
    NAME = 'collie'
    DEPS = [TvmApi, Sharpei, CollieDb]

    def __init__(self, env, components):
        super(Collie, self).__init__(env, components, binary_name='collie', custom_path='collie')
        self.__service = YplatformHttpService(
            env=env,
            name=self.NAME,
            binary_name='collie',
            custom_path='collie'
        )
        self.__pyremock_port = self.config[self.name]['pyremock_port']

    @staticmethod
    def gen_config(port_generator, config=None):
        result = YplatformComponent.gen_config(port_generator, config=config)
        result['pyremock_port'] = next(port_generator)
        return result

    def init_root(self):
        self.__service.init_root()
        write_collie_common_configs(self.__service)
        etc_recognizer_path = os.path.join(self.get_root(), 'etc', 'recognizer')
        helpers.mkdir_recursive(etc_recognizer_path)
        helpers.write2file(resource.find('recognizer/dict.dict'), os.path.join(etc_recognizer_path, 'dict.dict'))
        helpers.write2file(resource.find('recognizer/queryrec.dict'), os.path.join(etc_recognizer_path, 'queryrec.dict'))
        helpers.write2file(resource.find('recognizer/queryrec.weights'), os.path.join(etc_recognizer_path, 'queryrec.weights'))
        helpers.write2file(
            what=self.__service.format_config(resource.find('collie/config-base.yml')),
            path=os.path.join(self.__service.get_etc_path(), 'config-base.yml'),
        )
        helpers.write2file(
            what=self.__service.format_config(
                config=resource.find('collie/config-devpack.yml'),
                tvm_api_port=self.components[TvmApi].port,
                sharpei_port=(
                    self.components[Sharpei].webserver_port()
                    if Sharpei in self.components else 0
                ),
                pyremock_port=self.__pyremock_port,
            ),
            path=self.__service.get_config_path(),
        )

    def start(self):
        self.__service.start('pong')

    def info(self):
        base = super(Collie, self).info()
        base.update(self.__service.info())
        return base

    def pyremock_port(self):
        return self.__pyremock_port

    def api(self, uid=None, service_ticket=None, user_ticket=None):
        return CollieApi(
            location='http://localhost:%d' % self.webserver_port(),
            uid=uid,
            service_ticket=service_ticket,
            user_ticket=user_ticket,
        )

    def get_config_path(self):
        return self.__service.get_config_path()


class CollieApi(object):
    def __init__(self, location='http://localhost:12930', uid=None, service_ticket=None, user_ticket=None):
        self.location = location
        self.uid = uid
        self.service_ticket = service_ticket
        self.user_ticket = user_ticket

    @retrying.retry(stop_max_delay=10000)
    def ping(self, request_id):
        return requests.get(
            self.location + '/ping',
            headers=self.make_headers(request_id),
            timeout=1,
        )

    @retrying.retry(stop_max_delay=10000)
    def get_tags(self, request_id):
        return requests.get(
            self.location + '/v1/users/{uid}/tags'.format(uid=self.uid),
            headers=self.make_headers(request_id),
            timeout=5,
        )

    @retrying.retry(stop_max_delay=10000)
    def create_tag(self, name, request_id):
        return requests.post(
            self.location + '/v1/users/{uid}/tags'.format(uid=self.uid),
            json=dict(name=name),
            headers=self.make_headers(request_id),
            timeout=5,
        )

    @retrying.retry(stop_max_delay=10000)
    def remove_tag(self, tag_id, request_id):
        return requests.delete(
            self.location + '/v1/users/{uid}/tags/{tag_id}'.format(uid=self.uid, tag_id=tag_id),
            headers=self.make_headers(request_id),
            timeout=5,
        )

    @retrying.retry(stop_max_delay=10000)
    def get_contacts_with_tag(self, request_id, tag_id, offset=None, limit=None):
        params = {}
        if offset is not None:
            params['offset'] = offset
        if limit is not None:
            params['limit'] = limit

        return requests.get(
            self.location + '/v1/users/{uid}/tags/{tag_id}/contacts'.format(uid=self.uid, tag_id=tag_id),
            params=params,
            headers=self.make_headers(request_id),
            timeout=5
        )

    @retrying.retry(stop_max_delay=10000)
    def update_tag_name(self, name, tag_id, revision, request_id):
        return requests.put(
            self.location + '/v1/users/{uid}/tags/{tag_id}'.format(uid=self.uid, tag_id=tag_id),
            json=dict(name=name, revision=revision),
            headers=self.make_headers(request_id),
        )

    @retrying.retry(stop_max_delay=10000)
    def create_contacts(self, contacts, request_id):
        return requests.post(
            self.location + '/v1/users/{uid}/contacts'.format(uid=self.uid),
            json=contacts,
            headers=self.make_headers(request_id),
            timeout=5,
        )

    @retrying.retry(stop_max_delay=10000)
    def get_contacts(self, contact_ids, request_id, mixin=None, offset=None, limit=None, only_shared=None):
        params = {}
        if offset is not None:
            params['offset'] = offset
        if limit is not None:
            params['limit'] = limit
        if mixin is not None:
            params['mixin'] = mixin
        if only_shared is not None:
            params['only_shared'] = only_shared

        return requests.get(
            self.location + '/v1/users/{uid}/contacts/{contact_ids}'.format(
                uid=self.uid,
                contact_ids=urllib.quote(json.dumps(contact_ids)),
            ),
            params=params,
            headers=self.make_headers(request_id),
            timeout=5
        )

    @retrying.retry(stop_max_delay=10000)
    def remove_contacts(self, contact_ids, request_id):
        return requests.delete(
            self.location + '/v1/users/{uid}/contacts/{contact_ids}'.format(
                uid=self.uid,
                contact_ids=urllib.quote(json.dumps(contact_ids)),
            ),
            headers=self.make_headers(request_id),
            timeout=5,
        )

    @retrying.retry(stop_max_delay=10000)
    def get_emails(self, tag_ids, request_id):
        return requests.get(
            self.location + '/v1/users/{uid}/emails/{tag_ids}'.format(
                uid=self.uid,
                tag_ids=urllib.quote(json.dumps(tag_ids))
            ),
            params={'mixin': 'mixin'},
            headers=self.make_headers(request_id),
            timeout=5,
        )

    @retrying.retry(stop_max_delay=10000)
    def update_contacts(self, updated_contacts, request_id):
        return requests.put(
            self.location + '/v1/users/%d/contacts' % self.uid,
            json=updated_contacts,
            headers=self.make_headers(request_id),
            timeout=5,
        )

    @retrying.retry(stop_max_delay=10000)
    def get_contacts_location(self, request_id):
        return requests.get(
            self.location + '/v1/users/{uid}/location'.format(uid=self.uid),
            headers=self.make_headers(request_id),
            timeout=5,
        )

    @retrying.retry(stop_max_delay=10000)
    def carddav_delete(self, request_id, uri):
        return requests.post(
            self.location + '/v1/users/{uid}/carddav/{uri}/delete'.format(uid=self.uid, uri=uri),
            headers=self.make_headers(request_id),
            timeout=5
        )

    @retrying.retry(stop_max_delay=10000)
    def carddav_multiget(self, request_id, body):
        return requests.post(
            self.location + '/v1/users/{uid}/carddav/multiget'.format(uid=self.uid),
            headers=self.make_carddav_headers(request_id, 'application/xml'),
            data=body,
            timeout=5,
        )

    @retrying.retry(stop_max_delay=10000)
    def carddav_propfind(self, request_id):
        return requests.get(
            self.location + '/v1/users/{uid}/carddav/propfind'.format(uid=self.uid),
            headers=self.make_headers(request_id),
            timeout=5,
        )

    @retrying.retry(stop_max_delay=10000)
    def carddav_put(self, request_id, body, uri, etag):
        return requests.post(
            self.location + '/v1/users/{uid}/carddav/{uri}/put/{etag}'.format(
                uid=self.uid, uri=uri, etag=etag),
            headers=self.make_carddav_headers(request_id, 'text/plain'),
            data=body,
            timeout=5
        )

    @retrying.retry(stop_max_delay=10000)
    def get_contacts_count(self, request_id, mixin=None):
        params = {}
        if mixin is not None:
            params['mixin'] = mixin
        return requests.get(
            self.location + '/v1/users/{uid}/contacts/count'.format(uid=self.uid),
            headers=self.make_headers(request_id),
            timeout=5,
            params=params
        )

    @retrying.retry(stop_max_delay=10000)
    def add_emails(self, request_id, recipients):
        return requests.post(
            self.location + '/v1/users/{uid}/contacts/emails'.format(uid=self.uid),
            json=recipients,
            headers=self.make_headers(request_id),
            timeout=5
        )

    @retrying.retry(stop_max_delay=10000)
    def search_contacts(self, request_id, mixin=None, group=None):
        params = {'uid': '{uid}'.format(uid=self.uid)}
        if mixin is not None:
            params['mixin'] = mixin
        if group is not None:
            params['group'] = group

        return requests.get(
            self.location + '/v1/searchContacts',
            params=params,
            headers=self.make_headers(request_id),
            timeout=5
        )

    @retrying.retry(stop_max_delay=10000)
    def colabook_feed_addrdb_get(self, request_id, to):
        return requests.get(
            self.location + '/compat/colabook_feed_addrdb',
            params={'uid': '{uid}'.format(uid=self.uid), 'to': to},
            headers=self.make_headers(request_id),
            timeout=5
        )

    @retrying.retry(stop_max_delay=10000)
    def colabook_feed_addrdb_post(self, request_id, to):
        return requests.post(
            self.location + '/compat/colabook_feed_addrdb',
            params={'uid': '{uid}'.format(uid=self.uid)},
            headers=self.make_headers(request_id),
            data={'to': to},
            timeout=5
        )

    @retrying.retry(stop_max_delay=10000)
    def add_directory_event(self, request_id, data):
        return requests.post(
            self.location + '/v1/organizations/events',
            json=data,
            headers=self.make_headers(request_id),
            timeout=5
        )

    @retrying.retry(stop_max_delay=10000)
    def get_changes(self, request_id):
        return requests.get(
            self.location + '/v1/users/{uid}/changes'.format(uid=self.uid),
            headers=self.make_headers(request_id),
            timeout=5,
        )

    @retrying.retry(stop_max_delay=10000)
    def restore(self, revision, request_id):
        return requests.post(
            self.location + '/v1/users/{uid}/changes/{revision}'.format(uid=self.uid, revision=revision),
            headers=self.make_headers(request_id),
            timeout=5,
        )

    @retrying.retry(stop_max_delay=10000)
    def get_shared_lists(self, request_id):
        return requests.get(
            self.location + '/v1/users/{uid}/shared/lists'.format(uid=self.uid),
            headers=self.make_headers(request_id),
            timeout=5,
        )

    @retrying.retry(stop_max_delay=10000)
    def get_shared_contacts_count_from_list(self, request_id, list_id, shared_with_emails=None):
        params = {}
        if shared_with_emails is not None:
            params['shared_with_emails'] = shared_with_emails

        return requests.get(
            self.location + '/v1/users/{uid}/shared/lists/{list_id}/contacts/count'.format(uid=self.uid, list_id=list_id),
            params=params,
            headers=self.make_headers(request_id),
            timeout=5
        )

    @retrying.retry(stop_max_delay=10000)
    def get_shared_contacts_from_list(self, request_id, list_id, contact_ids, offset=None, limit=None,
                                      shared_with_emails=None):
        params = {}
        if offset is not None:
            params['offset'] = offset
        if limit is not None:
            params['limit'] = limit
        if shared_with_emails is not None:
            params['shared_with_emails'] = shared_with_emails

        return requests.get(
            self.location + '/v1/users/{uid}/shared/lists/{list_id}/contacts/{contact_ids}'.format(
                uid=self.uid, list_id=list_id, contact_ids=urllib.quote(json.dumps(contact_ids))),
            params=params,
            headers=self.make_headers(request_id),
            timeout=5
        )

    def make_headers(self, request_id):
        return {
            'X-Request-Id': request_id,
            'X-Ya-Service-Ticket': self.service_ticket,
            'X-Ya-User-Ticket': self.user_ticket,
        }

    def make_carddav_headers(self, request_id, content_type):
        headers = self.make_headers(request_id)
        headers['Content-Type'] = content_type
        return headers


class CollieDirectorySyncWorker(YplatformComponent):
    NAME = 'collie_directory_sync_worker'
    DEPS = [TvmApi, Sharpei, CollieDbForDirectorySync]

    def __init__(self, env, components):
        super(CollieDirectorySyncWorker, self).__init__(env, components, binary_name='collie', custom_path='collie')
        config = env.get_config()
        self.__components = components
        self.__state = read_state(config=config, component_name=self.NAME)
        self.__service = YplatformHttpService(
            env=env,
            name=self.NAME,
            binary_name='collie',
            custom_path='collie',
        )
        self.__pyremock_port = config[self.NAME]['pyremock_port']

    @property
    def state(self):
        return self.__state

    @staticmethod
    def gen_config(port_generator, config=None):
        result = YplatformComponent.gen_config(port_generator, config=config)
        result['pyremock_port'] = next(port_generator)
        return result

    def init_root(self):
        self.__service.init_root()
        write_collie_common_configs(self.__service)
        helpers.write2file(
            what=self.__service.format_config(
                config=resource.find('collie/config-directory-sync-worker-devpack.yml'),
                tvm_api_port=self.__components[TvmApi].port,
                sharpei_port=(
                    self.__components[Sharpei].webserver_port()
                    if Sharpei in self.__components else 0
                ),
                pyremock_port=self.__pyremock_port,
            ),
            path=self.__service.get_config_path(),
        )

    def start(self):
        self.__service.start(ping_response=None)

    def prepare_data(self):
        pass

    def pyremock_port(self):
        return self.__pyremock_port

    def get_config_path(self):
        return self.__service.get_config_path()


class CollieSyncWorker(YplatformComponent):
    NAME = 'collie_sync_worker'
    DEPS = [TvmApi, Sharpei, CollieDbForSync]

    def __init__(self, env, components):
        super(CollieSyncWorker, self).__init__(env, components, binary_name='collie', custom_path='collie')
        config = env.get_config()
        self.__components = components
        self.__service = YplatformHttpService(
            env=env,
            name=self.NAME,
            binary_name='collie',
            custom_path='collie',
        )
        self.__pyremock_port = config[self.NAME]['pyremock_port']

    @staticmethod
    def gen_config(port_generator, config=None):
        result = YplatformComponent.gen_config(port_generator, config=config)
        result['pyremock_port'] = next(port_generator)
        return result

    def init_root(self):
        self.__service.init_root()
        write_collie_common_configs(self.__service)
        helpers.write2file(
            what=self.__service.format_config(
                config=resource.find('collie/config-sync-devpack.yml'),
                tvm_api_port=self.__components[TvmApi].port,
                sharpei_port=(
                    self.__components[Sharpei].webserver_port()
                    if Sharpei in self.__components else 0
                ),
                pyremock_port=self.__pyremock_port,
            ),
            path=self.__service.get_config_path(),
        )

    def start(self):
        self.__service.start(ping_response=None)

    def prepare_data(self):
        pass

    def pyremock_port(self):
        return self.__pyremock_port

    def get_config_path(self):
        return self.__service.get_config_path()


def write_collie_common_configs(service):
    helpers.write2file(
        what=resource.find('collie/tvm_secret'),
        path=os.path.join(service.get_etc_path(), 'tvm_secret'),
    )
    helpers.write2file(
        what=resource.find('collie/contacts.sql'),
        path=os.path.join(service.get_etc_path(), 'contacts.sql'),
    )
    helpers.write2file(
        what=resource.find('collie/events_queue.sql'),
        path=os.path.join(service.get_etc_path(), 'events_queue.sql'),
    )
