// Copyright (c) 2017 Yandex LLC. All rights reserved.
// Use of this source code is governed by a MPL 2.0
// license that can be found in the LICENSE file.
// Author: Vladimir Skipor <skipor@yandex-team.ru>

package main

import (
	"bufio"
	"fmt"
	"io/ioutil"
	"log"
	"net/http"
	"os"
	"regexp"
	"strings"
	"time"

	"github.com/yandex/pandora/cli"
	"github.com/yandex/pandora/core"
	"github.com/yandex/pandora/core/aggregator/netsample"
	coreimport "github.com/yandex/pandora/core/import"
	"github.com/yandex/pandora/core/register"
)

var state = map[string]*user{}

// Ammo is structure of the bullits
type Ammo struct {
	Tag   string
	Count int
	UID   string
}

type user struct {
	CIDs []string
}

// Payload is structure for construct HTTP requests
type Payload struct {
	Method     string
	URI        string
	Data       string
	Assert     string
	Xrequestid string
}

type gunConfig struct {
	Target string `validate:"required"` // Configuration will fail, without target defined
}

// Sample is structure for aggregate the result of the shot
type Sample struct {
	URL              string
	ShootTimeSeconds float64
}

// Gun is structure for traffic generator
type Gun struct {
	// Configured on construction.
	client http.Client
	conf   gunConfig
	// Configured on Bind, before shooting
	aggr core.Aggregator // May be your custom Aggregator.
	core.GunDeps
}

func newGun(conf gunConfig) *Gun {
	return &Gun{conf: conf}
}

// Bind is the constructor for Gun
func (g *Gun) Bind(aggr core.Aggregator, deps core.GunDeps) error {
	tr := &http.Transport{
		MaxIdleConns:       1,
		IdleConnTimeout:    600 * time.Second,
		DisableCompression: true,
	}
	g.client = http.Client{Transport: tr} //keep-alive shooting
	g.aggr = aggr
	g.GunDeps = deps
	return nil
}

// Shoot is validator of the ammo file
func (g *Gun) Shoot(ammo core.Ammo) {
	customAmmo := ammo.(*Ammo) // Shoot will panic on unexpected ammo type. Panic cancels shooting.
	_, exist := state[customAmmo.UID]
	if exist {
		g.shoot(customAmmo)
	} else {
		fmt.Println("Wrong Ammo!")
	}
}

func (g *Gun) shoot(ammo *Ammo) {
	code := 0
	payload := g.genPayload(ammo)
	req := g.makeReq(payload)
	sample := netsample.Acquire(ammo.Tag)
	rs, err := g.client.Do(req)
	if err != nil {
		code = 0
	} else {
		code = rs.StatusCode
		if code == 200 {
			respBody, _ := ioutil.ReadAll(rs.Body)
			re := regexp.MustCompile(payload.Assert)
			if payload.Assert == "" || re.FindString(string(respBody)) == "" {
				code = rs.StatusCode
			} else {
				code = 314
				fmt.Println("error", payload.URI, payload.Xrequestid, rs.StatusCode, string(respBody))
			}
		} else {
			respBody, _ := ioutil.ReadAll(rs.Body)
			fmt.Println("fatal", payload.URI, payload.Xrequestid, rs.StatusCode, string(respBody))
		}
		_ = rs.Body.Close()
	}
	defer func() {
		sample.SetProtoCode(code)
		g.aggr.Report(sample)
		// remove vcard.rfc contact
		if ammo.Tag == "put_vcard" {
			req, _ = http.NewRequest("POST", strings.Join([]string{"http://", g.conf.Target, "/v1/users/", ammo.UID, "/carddav/ya-", ammo.UID, "/delete"}, ""), strings.NewReader(""))
			rs, err = g.client.Do(req)
			if err != nil {
				code = 0
				fmt.Println("DELETE ERROR! UID:" + ammo.UID)
			} else {
				code = rs.StatusCode
			}
		}
	}()
}

func initialState() {
	var uid string
	file, err := os.Open("./contacts.csv")
	if err != nil {
		log.Fatal(err)
	}
	defer func() { _ = file.Close() }()

	scanner := bufio.NewScanner(file)
	for scanner.Scan() {
		uid = strings.Split(scanner.Text(), ";")[0]
		state[uid] = &user{
			CIDs: strings.Split(strings.Split(scanner.Text(), ";")[1], ","),
		}
	}
}

func main() {

	initialState()
	fs := coreimport.GetFs()
	coreimport.Import(fs)
	// May not be imported, if you don't need http guns and etc.
	// Custom imports. Integrate your custom types into configuration system.
	coreimport.RegisterCustomJSONProvider("collie_provider", func() core.Ammo { return &Ammo{} })
	register.Gun("collie", newGun, func() gunConfig {
		return gunConfig{
			Target: "default target",
		}
	})

	cli.Run()
}
