#pragma once

#include "config.hpp"
#include "logic/db/add_directory_event_impl.hpp"
#include "logic/db/add_emails_impl.hpp"
#include "logic/db/carddav_delete_impl.hpp"
#include "logic/db/carddav_multiget_impl.hpp"
#include "logic/db/carddav_propfind_impl.hpp"
#include "logic/db/carddav_put_impl.hpp"
#include "logic/db/create_contacts_impl.hpp"
#include "logic/db/create_tag_impl.hpp"
#include "logic/db/get_changes_impl.hpp"
#include "logic/db/get_contacts_count_impl.hpp"
#include "logic/db/get_contacts_impl.hpp"
#include "logic/db/get_abook_format_contacts_impl.hpp"
#include "logic/db/get_contacts_with_tag_impl.hpp"
#include "logic/db/get_emails_impl.hpp"
#include "logic/db/get_shared_contacts_count_from_list_impl.hpp"
#include "logic/db/get_shared_contacts_from_list_impl.hpp"
#include "logic/db/get_shared_lists_impl.hpp"
#include "logic/db/get_tags_impl.hpp"
#include "logic/db/remove_contacts_impl.hpp"
#include "logic/db/remove_tag_impl.hpp"
#include "logic/db/restore_impl.hpp"
#include "logic/db/update_contacts_impl.hpp"
#include "logic/db/update_tag_impl.hpp"
#include "logic/db/sync_organizations_impl.hpp"
#include "server/handlers/abook_search_contacts_handler.hpp"
#include "server/handlers/abook_colabook_feed_addrdb_handler.hpp"
#include "server/handlers/add_directory_event_handler.hpp"
#include "server/handlers/add_emails_handler.hpp"
#include "server/handlers/carddav_delete_handler.hpp"
#include "server/handlers/carddav_multiget_handler.hpp"
#include "server/handlers/carddav_propfind_handler.hpp"
#include "server/handlers/carddav_put_handler.hpp"
#include "server/handlers/create_contacts_handler.hpp"
#include "server/handlers/create_tag_handler.hpp"
#include "server/handlers/get_changes_handler.hpp"
#include "server/handlers/get_contacts_count_handler.hpp"
#include "server/handlers/get_contacts_handler.hpp"
#include "server/handlers/get_contacts_with_tag_handler.hpp"
#include "server/handlers/get_emails_handler.hpp"
#include "server/handlers/get_shared_contacts_count_from_list_handler.hpp"
#include "server/handlers/get_shared_contacts_from_list_handler.hpp"
#include "server/handlers/get_shared_lists_handler.hpp"
#include "server/handlers/get_tags_handler.hpp"
#include "server/handlers/not_implemented.hpp"
#include "server/handlers/parameters.hpp"
#include "server/handlers/remove_contacts_handler.hpp"
#include "server/handlers/remove_tag_handler.hpp"
#include "server/handlers/restore_handler.hpp"
#include "server/handlers/tvm_guarded.hpp"
#include "server/handlers/update_contacts_handler.hpp"
#include "server/handlers/update_tag_handler.hpp"
#include "server/handlers/sync_organizations_handler.hpp"
#include "server/router/router.hpp"
#include "services/db/contacts/make_connection_provider.hpp"
#include "services/db/events_queue/make_connection_provider.hpp"
#include "services/utils.hpp"
#include "services/sheltie/sheltie_client_impl.hpp"
#include "services/settings/settings_client_impl.hpp"
#include "utils.hpp"

#include <src/services/directory/directory_client_impl.hpp>
#include <ymod_webserver/response.h>

namespace collie {
namespace detail {

using namespace server;

template <ymod_webserver::methods::http_method name>
constexpr auto method = router::method<
    std::integral_constant<ymod_webserver::methods::http_method, name>>;

template <class TvmGuard, class Pong>
auto makeRoutes(
    const TvmGuard& tvmGuard,
    const Pong& pong,
    const AddDirectoryEventHandler& addEvent,
    const CarddavDeleteHandler& carddavDelete,
    const CarddavMultigetHandler& carddavMultiget,
    const CarddavPropfindHandler& carddavPropfind,
    const CarddavPutHandler& carddavPut,
    const CreateContactsHandler& createContacts,
    const UpdateContactsHandler& updateContactsPost,
    const UpdateContactsHandler& updateContactsPut,
    const GetContactsHandler& getContacts,
    const RemoveContactsHandler& removeContacts,
    const GetEmailsHandler& getEmailsWithTags,
    const GetSharedListsHandler& getSharedLists,
    const GetSharedContactsCountFromListHandler& getSharedContactsCountFromList,
    const GetSharedContactsFromListHandler& getSharedContactsFromList,
    const GetTagsHandler& getTags,
    const CreateTagHandler& createTag,
    const UpdateTagHandler& updateTag,
    const RemoveTagHandler& removeTag,
    const GetContactsWithTagHandler& getContactsWithTag,
    const GetChangesHandler& getChanges,
    const RestoreHandler& restore,
    const GetContactsCountHandler& getContactsCount,
    const AddEmailsHandler& addEmails,
    const AbookColabookFeedAddrdbHandler& abookColabookFeedAddrdbGet,
    const AbookColabookFeedAddrdbHandler& abookColabookFeedAddrdbPost,
    const AbookSearchContactsHandler& abookSearchContacts,
    const SyncOrganizationsHandler& syncOrganizations
) {
    using namespace router::literals;
    using namespace ymod_webserver::methods;

    using router::middleware;
    using router::parameter;

    constexpr auto uid = parameter<Uid>;
    constexpr auto uri = parameter<Uri>;
    constexpr auto etag = parameter<Etag>;
    constexpr auto contactIds = parameter<ContactIds>;
    constexpr auto listId = parameter<ListId>;
    constexpr auto tagId = parameter<TagId>;
    constexpr auto tagIds = parameter<TagIds>;
    constexpr auto revision = parameter<Revision>;

    constexpr auto put = method<mth_put>;
    constexpr auto post = method<mth_post>;
    constexpr auto delete_ = method<mth_delete>;
    constexpr auto get = method<mth_get>;

    return "ping"_l / get(pong)
        | "sync_organizations"_l / get(syncOrganizations)
        | "v1"_l / (
            "users"_l / uid / middleware(server::makeTvmGuarded(tvmGuard, "/v1/users/")) / (
                "contacts"_l / (
                    "count"_l / (
                        get(getContactsCount)
                    )
                    | "emails"_l / (
                        post(addEmails)
                    )
                    | post(createContacts)
                    | put(updateContactsPut)
                    | contactIds / (
                        get(getContacts)
                        | delete_(removeContacts)
                    )
                )
                | "update_contacts"_l / (post(updateContactsPost))
                | "emails"_l / (
                    tagIds / get(getEmailsWithTags)
                )
                | "tags"_l / (
                    post(createTag)
                    | get(getTags)
                    | tagId / (
                        put(updateTag)
                        | delete_(removeTag)
                        | "contacts"_l / get(getContactsWithTag)
                    )
                )
                | "changes"_l / (
                    get(getChanges)
                    | revision
                        / post(restore)
                )
                | "carddav"_l / (
                    "multiget"_l / post(carddavMultiget)
                    | "propfind"_l / get(carddavPropfind)
                    | uri / (
                        "delete"_l / post(carddavDelete)
                        | "put"_l / etag / post(carddavPut)
                    )
                )
                | "shared"_l / (
                    "lists"_l / (
                        get(getSharedLists)
                        | listId / "contacts"_l / (
                            "count"_l / get(getSharedContactsCountFromList)
                            | contactIds / get(getSharedContactsFromList)
                        )
                    )
                )
            )
            | "organizations"_l / middleware(server::makeTvmGuarded(tvmGuard, "/v1/organizations/")) /
                "events"_l / post(addEvent)
            | "searchContacts"_l  / middleware(server::makeTvmGuarded(tvmGuard, "/v1/searchContacts/")) / get(abookSearchContacts)
        )
        | "compat"_l / "colabook_feed_addrdb"_l / middleware(server::makeTvmGuarded(tvmGuard,
                "/compat/colabook_feed_addrdb/")) / (
            get(abookColabookFeedAddrdbGet)
            | post(abookColabookFeedAddrdbPost)
        );
}

} // namespace detail

template <class TvmGuard>
auto makeApi(const Config& config, const std::shared_ptr<const TvmGuard>& tvmGuard) {
    using namespace detail;
    using namespace server;

    using ymod_webserver::http::stream_ptr;

    const auto pong = [] (const stream_ptr& stream, const auto& ...) {
        stream->result(ymod_webserver::codes::ok, "pong");
        return expected<void>();
    };

    const auto getTvm2Module = services::makeGetTvm2Module(config.tvmModule);
    const auto getHttpClient = services::makeGetHttpClient(config.services.httpClientModule);
    const auto getSheltieClusterClient = services::makeGetClusterClient("sheltie_cluster_client");
    const auto getSettingsClusterClient = services::makeGetClusterClient("settings_cluster_client");

    const auto sheltieClient = std::make_shared<const services::sheltie::SheltieClientImpl>(
        getSheltieClusterClient
    );

    const auto settingsClient = std::make_shared<const services::settings::SettingsClientImpl>(
        getSettingsClusterClient,
        services::makeGetTvm2Module("tvm")
    );

    const auto directoryClient = std::make_shared<const services::directory::DirectoryClientImpl>(
        config.services.directory,
        getHttpClient,
        getTvm2Module
    );

    const services::recognizer::RecognizerPtr recognizer { Recognizer::create(
        config.services.recognizer.languageDict.c_str(),
        config.services.recognizer.languageWeights.c_str(),
        config.services.recognizer.encodingDict.c_str())
    };

    using WithCheckIsUserExists = services::db::contacts::WithCheckIsUserExists;
    using MakeContactsConnectionProvider = services::db::contacts::MakeConnectionProvider<WithCheckIsUserExists>;
    using MakeEventsQueueConnectionProvider = services::db::events_queue::MakeConnectionProvider;

    const MakeContactsConnectionProvider makeContactsConnectionProvider(
        config.services.db.contacts, getHttpClient);

    const AddDirectoryEventHandler addEvent(
        std::make_shared<logic::db::AddDirectoryEventImpl<GetEventQueueModule>>(
                makeGetEventQueueModule(config.eventQueueModule))
    );

    const CarddavDeleteHandler carddavDelete(
        std::make_shared<logic::db::CarddavDeleteImpl<MakeContactsConnectionProvider>>(
                makeContactsConnectionProvider)
    );

    const CarddavMultigetHandler carddavMultiget(
        std::make_shared<logic::db::CarddavMultigetImpl<MakeContactsConnectionProvider>>(
                makeContactsConnectionProvider, sheltieClient)
    );

    const CarddavPropfindHandler carddavPropfind(
        std::make_shared<logic::db::CarddavPropfindImpl<MakeContactsConnectionProvider>>(
                makeContactsConnectionProvider)
    );

    const CarddavPutHandler carddavPut(
        std::make_shared<logic::db::CarddavPutImpl<MakeContactsConnectionProvider>>(
                makeContactsConnectionProvider, sheltieClient)
    );

    const CreateContactsHandler createContacts(
        std::make_shared<logic::db::CreateContactsImpl<MakeContactsConnectionProvider>>(
                makeContactsConnectionProvider)
    );

    const UpdateContactsHandler updateContactsPost(
        std::make_shared<logic::db::UpdateContactsImpl<MakeContactsConnectionProvider>>(
                makeContactsConnectionProvider)
    );

    const UpdateContactsHandler updateContactsPut(
        std::make_shared<logic::db::UpdateContactsImpl<MakeContactsConnectionProvider>>(
                makeContactsConnectionProvider)
    );

    const GetContactsHandler getContacts(
        std::make_shared<logic::db::GetContactsImpl<MakeContactsConnectionProvider>>(
                makeContactsConnectionProvider
        )
    );

    const RemoveContactsHandler removeContacts(
        std::make_shared<logic::db::RemoveContactsImpl<MakeContactsConnectionProvider>>(
                makeContactsConnectionProvider)
    );

    const GetEmailsHandler getEmailsWithTags(
        std::make_shared<logic::db::GetEmailsImpl<MakeContactsConnectionProvider>>(
                makeContactsConnectionProvider)
    );

    const CreateTagHandler createTag(
        std::make_shared<logic::db::CreateTagImpl<MakeContactsConnectionProvider>>(
                makeContactsConnectionProvider)
    );

    const GetTagsHandler getTags(
        std::make_shared<logic::db::GetTagsImpl<MakeContactsConnectionProvider>>(
                makeContactsConnectionProvider)
    );

    const GetSharedListsHandler getSharedLists(
        std::make_shared<logic::db::GetSharedListsImpl<MakeContactsConnectionProvider>>(
            makeContactsConnectionProvider)
    );

    const GetSharedContactsCountFromListHandler getSharedContactsCountFromList(
        std::make_shared<logic::db::GetSharedContactsCountFromListImpl<MakeContactsConnectionProvider>>(
                makeContactsConnectionProvider)
    );

    const GetSharedContactsFromListHandler getSharedContactsFromList(
        std::make_shared<logic::db::GetSharedContactsFromListImpl<MakeContactsConnectionProvider>>(
                makeContactsConnectionProvider)
    );

    const RemoveTagHandler removeTag(
        std::make_shared<logic::db::RemoveTagImpl<MakeContactsConnectionProvider>>(
                makeContactsConnectionProvider)
    );

    const UpdateTagHandler updateTag(
        std::make_shared<logic::db::UpdateTagImpl<MakeContactsConnectionProvider>>(
                makeContactsConnectionProvider)
    );

    const GetContactsWithTagHandler getContactsWithTag(
        std::make_shared<logic::db::GetContactsWithTagImpl<MakeContactsConnectionProvider>>(
                makeContactsConnectionProvider)
    );

    const GetChangesHandler getChanges(
        std::make_shared<logic::db::GetChangesImpl<MakeContactsConnectionProvider>>(
                makeContactsConnectionProvider)
    );

    const RestoreHandler restore(
        std::make_shared<logic::db::RestoreImpl<MakeContactsConnectionProvider>>(
                makeContactsConnectionProvider)
    );

    const GetContactsCountHandler getContactsCount(
        std::make_shared<logic::db::GetContactsCountImpl<MakeContactsConnectionProvider>>(
                makeContactsConnectionProvider
        )
    );

    const AddEmailsHandler addEmails(
        std::make_shared<logic::db::AddEmailsImpl<MakeContactsConnectionProvider>>(
                makeContactsConnectionProvider,
                settingsClient,
                recognizer)
    );

    const AbookColabookFeedAddrdbHandler abookColabookFeedAddrdbGet(
        std::make_shared<logic::db::AddEmailsImpl<MakeContactsConnectionProvider>>(
                makeContactsConnectionProvider,
                settingsClient,
                recognizer)
    );

    const AbookColabookFeedAddrdbHandler abookColabookFeedAddrdbPost(
        std::make_shared<logic::db::AddEmailsImpl<MakeContactsConnectionProvider>>(
                makeContactsConnectionProvider,
                settingsClient,
                recognizer)
    );

    const AbookSearchContactsHandler abookSearchContacts(
        std::make_shared<logic::db::GetAbookFormatContactsImpl<MakeContactsConnectionProvider>>(
                makeContactsConnectionProvider
        )
    );

    const MakeEventsQueueConnectionProvider makeEventsQueueConnectionProvider(
        config.services.db.eventsQueue, getHttpClient);

    const SyncOrganizationsHandler syncOrganizations(
        std::make_shared<logic::db::SyncOrganizationsImpl<MakeEventsQueueConnectionProvider>>(
                makeEventsQueueConnectionProvider,
                directoryClient)
    );

    return makeRoutes(
        tvmGuard,
        pong,
        addEvent,
        carddavDelete,
        carddavMultiget,
        carddavPropfind,
        carddavPut,
        createContacts,
        updateContactsPost,
        updateContactsPut,
        getContacts,
        removeContacts,
        getEmailsWithTags,
        getSharedLists,
        getSharedContactsCountFromList,
        getSharedContactsFromList,
        getTags,
        createTag,
        updateTag,
        removeTag,
        getContactsWithTag,
        getChanges,
        restore,
        getContactsCount,
        addEmails,
        abookColabookFeedAddrdbGet,
        abookColabookFeedAddrdbPost,
        abookSearchContacts,
        syncOrganizations
    );
}

} // namespace collie
