#include "module.hpp"
#include "config_reflection.hpp"
#include "run_sync_event_queue.hpp"

#include <pa/async.h>
#include <src/services/utils.hpp>
#include <yamail/data/deserialization/ptree.h>
#include <yplatform/module_registration.h>

namespace collie::directory_sync::event_queue {

Module::Module(boost::asio::io_context& io, const yplatform::ptree& configurationData)
        : io(io)
        , config(yamail::data::deserialization::fromPtree<Config>(configurationData))
        , syncEventQueueContext(std::make_shared<SyncEventQueueContext>())
        , timer(io) {
    if (!yplatform::find_reactor(configurationData.get<std::string>("reactor"))->plain()) {
        throw std::runtime_error{"collie_directory_event_queue is optimized for single-threaded reactors"};
    }

    if (!pa::async_profiler::is_init()) {
        const auto maxSize{1000000};
        const auto dumpSize{16000};
        pa::async_profiler::init(maxSize, dumpSize, config.profilerLogName);
    }
}

void Module::start() {
    syncTime = boost::posix_time::second_clock::universal_time();
    if (!config.syncAtStartup) {
        syncTime += boost::posix_time::seconds{config.syncIntervalInSeconds};
    }

    scheduleEventQueueSync();
}

void Module::stop() {
    timer.cancel();
    syncEventQueueContext->stop();
}

void Module::addEvent(logic::NewEvent newEvent) {
    boost::asio::post(io.get(), [=]{
        if (eventInfoByOrgId.find(newEvent.org_id) == eventInfoByOrgId.end()
            || eventInfoByOrgId[newEvent.org_id].revision <= newEvent.revision)
        {
            eventInfoByOrgId[newEvent.org_id].event = newEvent.event;
            eventInfoByOrgId[newEvent.org_id].revision = newEvent.revision;
        }
        ++eventInfoByOrgId[newEvent.org_id].count;
    });
}

void Module::scheduleEventQueueSync() {
    timer.expires_at(syncTime);
    timer.async_wait([&](const auto& error) {
        if (error) {
            YLOG_G(error) << "Directory event queue sync timer error " << error.value() << R"( (")" <<
                    error.message() << R"("))";
            return;
        }

        sync();
        syncTime += boost::posix_time::seconds{config.syncIntervalInSeconds};
        scheduleEventQueueSync();
    });
}

void Module::sync() {
    if (syncIsOngoing || eventInfoByOrgId.empty()) {
        return;
    }

    syncIsOngoing = true;

    using MakeEventsQueueConnectionProvider = services::db::events_queue::MakeConnectionProvider;
    const auto getHttpClient{services::makeGetHttpClient(config.services.httpClientModule)};
    const RunSyncEventQueue runSyncEventQueue{syncEventQueueContext, MakeEventsQueueConnectionProvider{
            config.services.db.eventsQueue, getHttpClient}};
    auto EventQueueSyncCoroutine{[=](auto yield) mutable {
        runSyncEventQueue(std::move(yield), config, eventInfoByOrgId);
        syncIsOngoing = false;
    }};

    boost::asio::spawn(io.get(), std::move(EventQueueSyncCoroutine), boost::coroutines::attributes(
            config.coroutineStackSize));
}

} // namespace collie::directory_sync::event_queue

DEFINE_SERVICE_OBJECT(collie::directory_sync::event_queue::Module)
