#pragma once

#include "error.hpp"
#include "sync_event_queue_context.hpp"

#include <src/logic/db/events_queue/add_directory_events.hpp>
#include <src/logic/interface/types/org_id.hpp>
#include <src/logic/interface/types/reflection/new_event.hpp>
#include <src/services/db/org_user_id.hpp>

#include <map>

namespace collie::directory_sync::event_queue {

using EventInfoByOrgId = std::map<logic::OrgId, logic::EventInfo>;

template<typename MakeEventsQueueConnectionProvider> class SyncEventQueue {
public:
    SyncEventQueue(
            const SyncEventQueueContext& syncEventQueueContext,
            const MakeEventsQueueConnectionProvider& makeEventsQueueConnectionProvider,
            const Config& config)
            : syncEventQueueContext(syncEventQueueContext)
            , makeEventsQueueConnectionProvider(makeEventsQueueConnectionProvider)
            , config(config) {
    }

    void operator()(EventInfoByOrgId& eventInfoByOrgId) const {
        const auto provider{makeEventsQueueConnectionProvider(syncEventQueueContext.getTaskContext())};
        for(auto element{eventInfoByOrgId.begin()}; element != eventInfoByOrgId.end(); ) {
            if (syncEventQueueContext.mustStop()) {
                LOGDOG_(syncEventQueueContext.logger(), error, log::error_code=error_code{Error::interrupted},
                        log::message="Directory event queue sync has been interrupted");
                return;
            }

            logic::EventInfo& elementValue = element->second;
            auto elementOrgId{element->first};
            const auto count{elementValue.count};
            auto result{logic::db::events_queue::addDirectoryEvents(provider, elementOrgId, elementValue)};
            if (!result) {
                LOGDOG_(syncEventQueueContext.logger(), error, log::error_code=std::move(result).error(),
                        log::org_id=elementOrgId, log::message="failed to add Directory events");
                ++element;
            } else {
                if (elementValue.count == count) {
                    element = eventInfoByOrgId.erase(element);
                } else {
                    elementValue.count -= count;
                    ++element;
                }
            }
        }
    }

private:
    const SyncEventQueueContext& syncEventQueueContext;
    const MakeEventsQueueConnectionProvider& makeEventsQueueConnectionProvider;
    const Config& config;
};

} // namespace collie::directory_sync::event_queue
