#include "prepare_contacts.hpp"

#include <src/utils.hpp>

#include <boost/range/algorithm/transform.hpp>

namespace collie::directory_sync {

using logic::DirectoryEntityId;

PrepareContacts::PrepareContacts(const Config& config, OrgId orgId, OrgInfo orgInfo)
        : config{config}
        , orgId{orgId}
        , orgInfo{std::move(orgInfo)} {
}

PrepareContacts::Contacts PrepareContacts::operator()(std::vector<DirectoryEntity> directoryEntities,
        Users users, Departments departments, Groups groups) const {
    const auto departmentsNames{makeDepartmentsNamesMap(departments)};
    Contacts result;
    auto existingDirectoryEntities{makeDirectoryEntitiesMap(directoryEntities)};
    fillPreparedContacts(departmentsNames, users, result, existingDirectoryEntities);
    fillPreparedContacts(departmentsNames, departments, result, existingDirectoryEntities);
    fillPreparedContacts(departmentsNames, groups, result, existingDirectoryEntities);

    for (auto& v : existingDirectoryEntities) {
        result.removedContacts.emplace_back(v.second);
        result.removedDirectoryEntities.emplace_back(std::move(v.first));
    }

    return result;
}

PrepareContacts::DirectoryEntitiesMap PrepareContacts::makeDirectoryEntitiesMap(
        const std::vector<DirectoryEntity>& entities) const {
    DirectoryEntitiesMap result;
    boost::transform(entities, std::inserter(result, result.end()), [](const auto& entity){
            return std::make_pair(DirectoryEntityIdAndType{entity.id, entity.type}, entity.contactId);});
    return result;
}

PrepareContacts::DepartmentsNamesMap PrepareContacts::makeDepartmentsNamesMap(
        const Departments& values) const {
    std::map<std::int64_t, std::string> result;
    for (const auto& v : values) {
        if (v.name) {
            result.emplace(v.id, *v.name);
        }
    }

    return result;
}

void PrepareContacts::preprocessContacts(std::optional<DirectoryContacts>& contacts) const {
    auto filter{[](const auto& contact){return (!contact.type) || (!contact.value);}};
    preprocessRange(contacts, std::move(filter));
}

Name PrepareContacts::makeName(PrepareContacts::DirectoryName value) const {
    Name result;
    result.first = std::move(value.first);
    result.last = std::move(value.last);
    return result;
}

template<typename Element, typename Transformer> std::optional<std::vector<Element>>
        PrepareContacts::makeFromContacts(const std::vector<std::string>& pattern,
                DirectoryContacts& contacts, Transformer transformer) const {
    auto filter{[&](const auto& contact){return isInRange(pattern, *contact.type);}};
    std::vector<Element> elements;
    boost::copy(contacts | boost::adaptors::filtered(std::move(filter)) | boost::adaptors::transformed(
            std::move(transformer)), std::back_inserter(elements));
    return elements.empty() ? std::optional<std::vector<Element>>{} : std::move(elements);
}

std::optional<std::vector<Email>> PrepareContacts::makeEmails(DirectoryContacts& contacts) const {
    auto transformer{[](auto& contact){return Email{std::move(contact.value), {}, {}};}};
    std::string contactType{"email"};
    return makeFromContacts<Email>({contactType}, contacts, std::move(transformer));
}

std::optional<std::vector<InstantMessenger>> PrepareContacts::makeInstantMessengers(
        DirectoryContacts& contacts) const {
    const auto instantMessengers{yplatform::util::split_unique(config.services.directory.instantMessengers,
            separator)};
    auto transformer{[](auto& contact){return InstantMessenger{{}, {}, std::move(contact.value), std::move(
            contact.type)};}};
    return makeFromContacts<InstantMessenger>(instantMessengers, contacts, std::move(transformer));
}

std::optional<std::vector<SocialProfile>> PrepareContacts::makeSocialProfiles(
        DirectoryContacts& contacts) const {
    const auto socialProfiles{yplatform::util::split_unique(config.services.directory.socialProfiles,
            separator)};
    auto transformer{[](auto& contact){return SocialProfile{std::move(contact.value), {{std::move(
            *contact.type)}}, {}};}};
    return makeFromContacts<SocialProfile>(socialProfiles, contacts, std::move(transformer));
}

std::optional<std::vector<TelephoneNumber>> PrepareContacts::makeTelephoneNumbers(
        DirectoryContacts& contacts) const {
    auto transformer{[](auto& contact){return TelephoneNumber{std::move(contact.value), {}, {}, {}};}};
    std::string contactType{"phone"};
    return makeFromContacts<TelephoneNumber>({contactType}, contacts, std::move(transformer));
}

std::optional<std::vector<Website>> PrepareContacts::makeWebsites(DirectoryContacts& contacts) const {
    auto transformer{[](auto& contact){return Website{std::move(contact.value), {}, {}};}};
    std::string contactType{"site"};
    return makeFromContacts<Website>({contactType}, contacts, std::move(transformer));
}

Organization PrepareContacts::makeOrganization(
        std::optional<std::string> position,
        const std::optional<std::int64_t>& departmentId,
        const PrepareContacts::DepartmentsNamesMap& departmentsNames,
        std::optional<std::string> summary) const {
    Organization result;
    if (position) {
        result.title = std::move(*position);
    }

    if (departmentId) {
        const auto name = departmentsNames.find(*departmentId);
        if (name != departmentsNames.end()) {
            result.department = std::string(name->second.begin(), name->second.end());
        }
    }

    result.summary = std::move(summary);

    return result;
}

DirectoryEntry PrepareContacts::makeDirectoryEntry(std::int64_t id, std::string type) const {
    return {{orgId}, orgInfo.name, {id}, {{std::move(type)}}};
}

Vcard PrepareContacts::makeVcard(User value,
        const PrepareContacts::DepartmentsNamesMap& departmentsNames) const {
    Vcard result;
    if (value.name) {
        result.names = std::vector(1, makeName(std::move(*value.name)));
    }

    std::string directoryEntryType{"user"};
    result.directory_entries = {{makeDirectoryEntry(value.id, std::move(directoryEntryType))}};
    if (value.birthday) {
        result.events = std::vector(1, makeBirthdayEvent(std::move(*value.birthday)));
    }

    preprocessContacts(value.contacts);
    if (value.contacts) {
        result.emails = makeEmails(*value.contacts);
        result.instant_messengers = makeInstantMessengers(*value.contacts);
        result.social_profiles = makeSocialProfiles(*value.contacts);
        result.telephone_numbers = makeTelephoneNumbers(*value.contacts);
        result.websites = makeWebsites(*value.contacts);
    }

    if (value.position || value.department) {
        std::optional<std::int64_t> id;
        if (value.department) {
            id = value.department->id;
        }
        result.organizations = {{makeOrganization(std::move(value.position), id, departmentsNames, {})}};
    }

    return result;
}

Name PrepareContacts::makeName(std::string value) const {
    Name result;
    result.first = std::move(value);
    return result;
}

Email PrepareContacts::makeEmail(std::string value) const {
    Email result;
    result.email = std::move(value);
    return result;
}

Vcard PrepareContacts::makeVcard(Department value,
        const PrepareContacts::DepartmentsNamesMap& departmentsNames) const {
    Vcard result;
    if (value.name) {
        result.names = std::vector(1, makeName(std::move(*value.name)));
    }

    if (value.email) {
        result.emails = std::vector(1, makeEmail(std::move(*value.email)));
    }

    std::string directoryEntryType{"department"};
    result.directory_entries = {{makeDirectoryEntry(value.id, std::move(directoryEntryType))}};
    result.description = value.description;
    if (value.parent && value.parent->id) {
        result.organizations = {{makeOrganization({}, value.parent->id, departmentsNames,
                std::move(value.description))}};
    }

    return result;
}

Vcard PrepareContacts::makeVcard(Group value,
        const PrepareContacts::DepartmentsNamesMap& departmentsNames) const {
    Vcard result;
    if (value.name) {
        result.names = std::vector(1, makeName(std::move(*value.name)));
    }

    if (value.email) {
        result.emails = std::vector(1, makeEmail(std::move(*value.email)));
    }

    std::string directoryEntryType{"group"};
    result.directory_entries = {{makeDirectoryEntry(value.id, std::move(directoryEntryType))}};
    result.description = value.description;
    result.organizations = {{makeOrganization({}, {}, departmentsNames, std::move(value.description))}};
    return result;
}

template<typename Entity> NewContact PrepareContacts::makeNewContact(Entity&& entity,
        const PrepareContacts::DepartmentsNamesMap& departmentsNames) const {
    NewContact result;
    result.vcard = makeVcard(std::forward<Entity>(entity), departmentsNames);
    return result;
}

template<typename Entity> UpdatedContact PrepareContacts::makeUpdatedContact(ContactId contactId,
        Entity&& entity, const PrepareContacts::DepartmentsNamesMap& departmentsNames) const {
    UpdatedContact result;
    result.contact_id = contactId;
    result.vcard = makeVcard(std::forward<Entity>(entity), departmentsNames);
    return result;
}

DirectoryEntityType PrepareContacts::getEntityType(const Department&) const {
    return DirectoryEntityType::department;
}

DirectoryEntityType PrepareContacts::getEntityType(const Group&) const {
    return DirectoryEntityType::group;
}

DirectoryEntityType PrepareContacts::getEntityType(const User&) const {
    return DirectoryEntityType::user;
}

DirectoryEntity PrepareContacts::makeDirectoryEntity(const DirectoryEntityIdAndType& value) const {
    DirectoryEntity result;
    result.id = value.id;
    result.type = value.type;
    result.contactId = ContactId {};
    return result;
}

template<typename Entity> void PrepareContacts::fillPreparedContacts(
        const PrepareContacts::DepartmentsNamesMap& departmentsNames,
        std::vector<Entity>& entities, PrepareContacts::Contacts& result,
        PrepareContacts::DirectoryEntitiesMap& existingDirectoryEntities) const {
    for (auto& entity : entities) {
        const DirectoryEntityIdAndType entityIdAndType{DirectoryEntityId{entity.id}, getEntityType(entity)};
        const auto it = existingDirectoryEntities.find(entityIdAndType);
        if (it == existingDirectoryEntities.end()) {
            result.newDirectoryEntities.push_back(makeDirectoryEntity(entityIdAndType));
            result.newContacts.push_back(makeNewContact(std::move(entity), departmentsNames));
        } else {
            result.updatedContacts.push_back(
                makeUpdatedContact(it->second, std::move(entity), departmentsNames));
            existingDirectoryEntities.erase(it);
        }
    }
}

}
