#pragma once

#include "config.hpp"

#include <src/logic/interface/types/directory_entity.hpp>
#include <src/logic/interface/types/directory_entity_id_and_type.hpp>
#include <src/logic/interface/types/new_contact.hpp>
#include <src/logic/interface/types/updated_contact.hpp>
#include <src/services/directory/types.hpp>

namespace collie::directory_sync {

using logic::ContactId;
using logic::DirectoryEntity;
using logic::DirectoryEntityIdAndType;
using logic::DirectoryEntityType;
using logic::DirectoryEntry;
using logic::Email;
using logic::InstantMessenger;
using logic::Name;
using logic::NewContact;
using logic::Organization;
using logic::SocialProfile;
using logic::TelephoneNumber;
using logic::UpdatedContact;
using logic::Vcard;
using logic::Website;
using services::directory::Department;
using services::directory::Departments;
using services::directory::Group;
using services::directory::Groups;
using services::directory::OrgInfo;
using services::directory::OrgId;
using services::directory::User;
using services::directory::Users;

using DirectoryContacts = std::vector<collie::services::directory::Contact>;

class PrepareContacts final {
public:
    PrepareContacts(const Config& config, OrgId orgId, OrgInfo orgInfo);

    struct Contacts {
        std::vector<NewContact> newContacts;
        std::vector<UpdatedContact> updatedContacts;
        std::vector<ContactId> removedContacts;
        std::vector<DirectoryEntity> newDirectoryEntities;
        std::vector<DirectoryEntityIdAndType> removedDirectoryEntities;
    };

    Contacts operator()(std::vector<DirectoryEntity> directoryEntities, Users users, Departments departments,
            Groups groups) const;
private:
    using DirectoryEntitiesMap = std::map<DirectoryEntityIdAndType, std::int64_t>;
    DirectoryEntitiesMap makeDirectoryEntitiesMap(const std::vector<DirectoryEntity>& values) const;

    using DepartmentsNamesMap = std::map<std::int64_t, std::string>;
    DepartmentsNamesMap makeDepartmentsNamesMap(const Departments& values) const;

    void preprocessContacts(std::optional<DirectoryContacts>& contacts) const;

    using DirectoryName = collie::services::directory::Name;
    Name makeName(DirectoryName value) const;

    template<typename Element, typename Transformer> std::optional<std::vector<Element>> makeFromContacts(
            const std::vector<std::string>& pattern,
            DirectoryContacts& contacts,
            Transformer transformer) const;
    std::optional<std::vector<Email>> makeEmails(DirectoryContacts& contacts) const;
    std::optional<std::vector<InstantMessenger>> makeInstantMessengers(DirectoryContacts& contacts) const;
    std::optional<std::vector<SocialProfile>> makeSocialProfiles(DirectoryContacts& contacts) const;
    std::optional<std::vector<TelephoneNumber>> makeTelephoneNumbers(DirectoryContacts& contacts) const;
    std::optional<std::vector<Website>> makeWebsites(DirectoryContacts& contacts) const;
    Organization makeOrganization(
            std::optional<std::string> position,
            const std::optional<std::int64_t>& departmentId,
            const DepartmentsNamesMap& departmentsNames,
            std::optional<std::string> summary) const;
    DirectoryEntry makeDirectoryEntry(std::int64_t id, std::string type) const;
    Vcard makeVcard(User value, const DepartmentsNamesMap& departmentsNames) const;
    Name makeName(std::string value) const;
    Email makeEmail(std::string value) const;
    Vcard makeVcard(Department value, const DepartmentsNamesMap& departmentsNames) const;
    Vcard makeVcard(Group value, const DepartmentsNamesMap& departmentsNames) const;
    template<typename Entity> NewContact makeNewContact(Entity&& entity,
            const DepartmentsNamesMap& departmentsNames) const;
    template<typename Entity> UpdatedContact makeUpdatedContact(ContactId contactId, Entity&& entity,
            const DepartmentsNamesMap& departmentsNames) const;
    DirectoryEntityType getEntityType(const Department&) const;
    DirectoryEntityType getEntityType(const Group&) const;
    DirectoryEntityType getEntityType(const User&) const;
    DirectoryEntity makeDirectoryEntity(const DirectoryEntityIdAndType& value) const;
    template<typename Entity> void fillPreparedContacts(
            const DepartmentsNamesMap& departmentsNames,
            std::vector<Entity>& entities, Contacts& result,
            DirectoryEntitiesMap& existingDirectoryEntities) const;

    const Config& config;
    const OrgId orgId;
    const OrgInfo orgInfo;
    const std::string separator{","};
};

}
