#pragma once

#include "config.hpp"
#include "error.hpp"
#include "prepare_contacts.hpp"
#include "worker_context.hpp"
#include "types/user_to_sync.hpp"

#include <src/get_io_context.hpp>

#include <src/logic/interface/types/reflection/new_event.hpp>
#include <src/logic/db/contacts/acquire_revision.hpp>
#include <src/logic/db/contacts/create_contacts.hpp>
#include <src/logic/db/contacts/create_contacts_user.hpp>
#include <src/logic/db/contacts/create_directory_entities.hpp>
#include <src/logic/db/contacts/get_directory_entities.hpp>
#include <src/logic/db/contacts/get_shared_lists.hpp>
#include <src/logic/db/contacts/purge_contacts_user.hpp>
#include <src/logic/db/contacts/remove_contacts_completely.hpp>
#include <src/logic/db/contacts/remove_directory_entities.hpp>
#include <src/logic/db/contacts/subscribe_to_contacts.hpp>
#include <src/logic/db/contacts/unsubscribe_from_contacts.hpp>
#include <src/logic/db/contacts/update_contacts.hpp>

#include <src/services/db/begin.hpp>
#include <src/services/db/commit.hpp>
#include <src/services/db/get_connection.hpp>
#include <src/services/db/passport_user_id.hpp>
#include <src/services/db/request.hpp>
#include <src/services/db/events_queue/get_user_to_sync.hpp>
#include <src/services/db/events_queue/complete_directory_organization_removal.hpp>
#include <src/services/db/events_queue/complete_sync_directory_event.hpp>
#include <src/services/db/contacts/query.hpp>
#include <src/services/directory/error.hpp>
#include <src/services/directory/directory_client.hpp>
#include <src/services/directory/utils.hpp>

#include <boost/range/adaptor/filtered.hpp>
#include <boost/range/adaptor/transformed.hpp>
#include <boost/range/algorithm/copy.hpp>
#include <boost/uuid/random_generator.hpp>
#include <boost/uuid/uuid_io.hpp>

namespace collie::directory_sync {

using services::directory::DirectoryClientPtr;
using services::db::OrgUserId;
using services::db::PassportUserId;
using services::db::getConnection;
using services::db::events_queue::DirectoryRevision;
using services::db::events_queue::DirectoryEventId;
using services::directory::Event;
using services::directory::Departments;
using services::directory::Groups;
using services::directory::OrgId;
using services::directory::Users;

enum class OrganizationRemovalResult {
    Success,
    Failure
};

template <
    class MakeEventsQueueConnectionProvider,
    class MakeConnectionProviderWithCheckUserExists,
    class MakeConnectionProviderWithoutCheckUserExists>
class Worker {
public:
    Worker(MakeEventsQueueConnectionProvider makeEventsQueueConnectionProvider,
            MakeConnectionProviderWithCheckUserExists makeConnProviderWithCheckUserExists,
            MakeConnectionProviderWithoutCheckUserExists makeConnProviderWithoutCheckUserExists,
            DirectoryClientPtr directoryClient,
            const Config& config)
        : makeEventsQueueConnectionProvider(std::move(makeEventsQueueConnectionProvider))
        , makeConnProviderWithCheckUserExists(std::move(makeConnProviderWithCheckUserExists))
        , makeConnProviderWithoutCheckUserExists(std::move(makeConnProviderWithoutCheckUserExists))
        , directory(directoryClient)
        , config(config) {
    }

    void operator ()(WorkerContextPtr context) const {
        boost::uuids::random_generator requestIdGenerator;
        const auto eventsQueue = makeEventsQueueConnectionProvider(context->sub());

        while (!context->shouldStop()) {
            boost::asio::post(context->yield());
            const auto runContext = boost::make_shared<TaskContext>(
                context->sub()->uniq_id(),
                boost::uuids::to_string(requestIdGenerator()),
                context->sub()->userIp(),
                context->sub()->clientType(),
                context->yield());
            process(runContext, eventsQueue)
                .catch_error([&] (auto&& ec) { logError(runContext, std::move(ec)); });
        }
    }

private:
    struct ProcessEventResult {
        DirectoryRevision syncedRevision;
        DirectoryEventId syncedEventId;
    };

    template<typename EventsQueue> expected<void> process(TaskContextPtr context,
            EventsQueue eventsQueue) const {
        static_assert(services::db::ConnectionProvider<EventsQueue>);

        using services::db::begin;
        using services::db::events_queue::GetUserToSync;
        using services::db::events_queue::CompleteSyncDirectoryEvent;
        using services::db::events_queue::CompleteDirectoryOrganizationRemoval;
        using DirectoryError = collie::services::directory::Error;

        const auto sync{[&](auto&& eventsQueueTransaction) {
            const GetUserToSync getUserToSync(eventsQueue);

            const auto processUser{[&](auto&& user) {

                LOGDOG_(context->logger(), notice, log::user_id=user.user_id,
                        log::message="start organization sync");

                auto completeEventSync{[&](auto&& result) {
                    CompleteSyncDirectoryEvent completeSyncDirectoryEvent{eventsQueue};
                    return completeSyncDirectoryEvent(eventsQueueTransaction, {user.user_id,
                            result.syncedRevision, result.syncedEventId});
                }};

                const auto removeOrganization{[&] {
                    return removeOrganizationImpl(
                        context,
                        makeConnProviderWithCheckUserExists(context, OrgUserId{user.user_id}));
                }};

                auto completeOrganizationRemoval{[&](auto&& result) {
                    if (result == OrganizationRemovalResult::Failure) {
                        return completeEventSync(ProcessEventResult{DirectoryRevision{user.synced_revision},
                            DirectoryEventId{user.synced_event_id}});
                    }
                    CompleteDirectoryOrganizationRemoval completeDirectoryOrganizationRemoval{eventsQueue};
                    return completeDirectoryOrganizationRemoval(eventsQueueTransaction, {user.user_id});
                }};

                auto event{getLastEvent(context, user)};
                if (!event) {
                    if (event.error() == DirectoryError::organizationDeleted) {
                        LOGDOG_(context->logger(), notice, log::user_id=user.user_id,
                            log::message="start organization removal");
                        return removeOrganization().bind(completeOrganizationRemoval);
                    }
                    LOGDOG_(context->logger(), error, log::error_code=std::move(event).error(), log::user_id=user.user_id,
                            log::message="failed to get last directory event");
                    return completeEventSync(ProcessEventResult{DirectoryRevision{user.synced_revision},
                            DirectoryEventId{user.synced_event_id}});
                }

                const auto syncEvent{[&] {
                    Event syncedEvent{{}, {}, {}, user.synced_event_id, user.synced_revision};
                    if (*event && eventLessByRevisionAndId(syncedEvent, **event)) {
                        return syncEventImpl(
                                context,
                                makeConnProviderWithCheckUserExists(context, OrgUserId{user.user_id}),
                                makeConnProviderWithoutCheckUserExists(context, OrgUserId{user.user_id}),
                                OrgId{user.user_id},
                                std::move(**event));
                    } else {
                        LOGDOG_(context->logger(), notice, log::user_id=user.user_id,
                                log::message="sync not required");
                        return make_expected(ProcessEventResult{DirectoryRevision{user.synced_revision},
                                DirectoryEventId{user.synced_event_id}});
                    }
                }};

                return syncEvent().bind(std::move(completeEventSync));
            }};

            const auto toUserType{[&](auto&& user) {
                if (!user) {
                    return make_expected_from_error<UserToSync>(error_code(Error::noUserToSync));
                }

                auto eventType = logic::makeEventType((user->event_type).has_value() ? (user->event_type).value().get() : "organization_updated");
                return make_expected(UserToSync{user->user_id, user->synced_event_id, user->synced_revision, eventType});
            }};

            const auto returnTransaction{[&]{return std::move(eventsQueueTransaction);}};
            return getUserToSync(eventsQueueTransaction, static_cast<std::int64_t>
                    (config.syncIntervalInMinutes)).bind(toUserType).bind(processUser).bind(returnTransaction);
        }};

        const auto commit{[&](auto&& eventsQueueTransaction){return services::db::commit(
                eventsQueue, std::move(eventsQueueTransaction));}};
        const auto ignoreResult{[](auto&&){}};
        return begin(eventsQueue).bind(sync).bind(commit).bind(ignoreResult);
    }

    expected<std::optional<Event>> getLastEvent(TaskContextPtr context, const UserToSync& user) const {
        services::directory::GetLastEventParams params;
        params.orgId = user.user_id;
        params.syncedRevision = user.synced_revision;
        return directory->getLastEvent(context, params);
    }

    template <class ProviderWithCheckUserExists, class ProviderWithoutCheckUserExists>
    expected<ProcessEventResult> syncEventImpl(
            TaskContextPtr context,
            ProviderWithCheckUserExists providerWithCheckUserExists,
            ProviderWithoutCheckUserExists providerWithoutCheckUserExists,
            OrgId orgId,
            Event event) const {
        using services::db::begin;
        using services::db::request;
        using services::db::commit;
        using logic::UpdatedContacts;

        namespace contacts = logic::db::contacts;

        static_assert(services::db::ConnectionProvider<ProviderWithCheckUserExists>);
        static_assert(services::db::ConnectionProvider<ProviderWithoutCheckUserExists>);

        auto orgInfo{directory->getOrgInfo(context, orgId)};
        if (!orgInfo) {
            return make_unexpected(std::move(orgInfo.error()));
        }

        auto users{directory->getUsers(context, orgId)};
        if (!users) {
            return make_unexpected(std::move(users.error()));
        }

        auto departments{directory->getDepartments(context, orgId)};
        if (!departments) {
            return make_unexpected(std::move(departments.error()));
        }

        auto groups{directory->getGroups(context, orgId)};
        if (!groups) {
            return make_unexpected(std::move(groups.error()));
        }

        const auto sync{[&](auto&& transaction) {
            const auto getDirectoryEntities{[&](auto&&){return contacts::getDirectoryEntities(transaction);}};
            const auto prepareContacts{[&](auto&& directoryEntities) {
                return PrepareContacts{config, orgId, std::move(*orgInfo)}(
                        std::move(directoryEntities), *users, std::move(*departments), std::move(*groups));
            }};

            const auto updateContacts{[&](auto&& preparedContacts) {
                if (!preparedContacts.removedDirectoryEntities.empty()) {
                    const auto removedDirectoryEntities{contacts::removeDirectoryEntities(
                            std::move(preparedContacts.removedDirectoryEntities), transaction)};
                    if (!removedDirectoryEntities) {
                        return make_expected_from_error<void>(std::move(removedDirectoryEntities.error()));
                    }
                }

                if (!preparedContacts.removedContacts.empty()) {
                    const auto removed{contacts::removeContactsCompletely(transaction,
                            preparedContacts.removedContacts, config.chunkSizeForModifyingOp)};
                    if (!removed) {
                        return make_expected_from_error<void>(std::move(removed.error()));
                    }
                }

                if (!preparedContacts.updatedContacts.empty()) {
                    UpdatedContacts updatedContacts;
                    updatedContacts.updated_contacts = std::move(preparedContacts.updatedContacts);
                    const auto updated{contacts::updateContacts(std::move(updatedContacts), transaction,
                            config.chunkSizeForModifyingOp)};
                    if (!updated) {
                        return make_expected_from_error<void>(std::move(updated.error()));
                    }
                }

                if (!preparedContacts.newContacts.empty()) {
                    auto saveContactIds{[&](const auto& createdContacts) {
                        for (auto i{0u}; i < createdContacts.contact_ids.size(); ++i) {
                            preparedContacts.newDirectoryEntities[i].contactId = createdContacts.
                                    contact_ids[i];
                        }
                    }};

                    const auto created{contacts::createContacts(std::move(preparedContacts.newContacts),
                            transaction, config.chunkSizeForModifyingOp).bind(std::move(saveContactIds))};

                    if (!created) {
                        return make_expected_from_error<void>(std::move(created.error()));
                    }
                }

                if (!preparedContacts.newDirectoryEntities.empty()) {
                    const auto createdDirectoryEntities{contacts::createDirectoryEntities(
                            std::move(preparedContacts.newDirectoryEntities), transaction)};
                    if (!createdDirectoryEntities) {
                        return make_expected_from_error<void>(std::move(createdDirectoryEntities.error()));
                    }
                }

                return make_expected();
            }};

            auto shareLists{[&]{
                auto sharedLists{contacts::getSharedLists(transaction)};
                if (!sharedLists) {
                    return make_expected_from_error<void>(std::move(sharedLists).error());
                }

                auto orgListId{contacts::getDefaultListId(transaction)};
                if (!orgListId) {
                    return make_expected_from_error<void>(std::move(orgListId).error());
                }

                const auto filter{[&](const auto& list){return list.list_id == *orgListId;}};
                const auto listTransformer{[&](const auto& list){return list.client_user_id;}};
                std::vector<std::int64_t> existingUserIds;
                boost::copy(*sharedLists | boost::adaptors::filtered(std::move(filter)) | boost::adaptors::
                        transformed(std::move(listTransformer)), std::back_inserter(existingUserIds));
                if (!std::is_sorted(existingUserIds.begin(), existingUserIds.end())) {
                    std::sort(existingUserIds.begin(), existingUserIds.end());
                }

                const auto userTransformer{[&](const auto& user){return user.id;}};
                std::vector<std::int64_t> actualUserIds;
                boost::copy(*users | boost::adaptors::transformed(std::move(userTransformer)),
                        std::back_inserter(actualUserIds));
                if (!std::is_sorted(actualUserIds.begin(), actualUserIds.end())) {
                    std::sort(actualUserIds.begin(), actualUserIds.end());
                }

                std::vector<std::int64_t> userIdsToUnsubscribe;
                std::set_difference(existingUserIds.begin(), existingUserIds.end(), actualUserIds.begin(),
                        actualUserIds.end(), std::back_inserter(userIdsToUnsubscribe));

                std::vector<std::int64_t> userIdsToSubscribe;
                std::set_difference(actualUserIds.begin(), actualUserIds.end(), existingUserIds.begin(),
                        existingUserIds.end(), std::back_inserter(userIdsToSubscribe));

                auto unsubscribe{[&](auto&& userProvider){return contacts::unsubscribeFromContacts(
                        transaction, std::move(userProvider), *orgListId);}};
                manageSubscriptions(context, std::move(userIdsToUnsubscribe), std::move(unsubscribe));

                const std::string clientListName{"org_contacts_" + std::to_string(orgId)};
                auto subscribe{[&](auto&& userProvider){return contacts::subscribeToContacts(
                        transaction, std::move(userProvider), *orgListId, clientListName);}};
                manageSubscriptions(context, std::move(userIdsToSubscribe), std::move(subscribe));

                return make_expected();
            }};

            const auto commit{[&]{return services::db::commit(providerWithCheckUserExists,
                    std::move(transaction)).bind([](auto &&){});}};

            return contacts::acquireRevision(transaction).
                bind(getDirectoryEntities).
                bind(prepareContacts).
                bind(updateContacts).
                bind(shareLists).
                bind(commit).
                bind([&]{return expected(ProcessEventResult{
                        DirectoryRevision{event.revision}, DirectoryEventId{event.id}});});
        }};

        auto connection = getConnection(providerWithCheckUserExists);
        if (!connection) {
            auto ec = connection.error();
            if (ec != logic::Error::userNotFound) {
                return make_unexpected(ec);
            }

            auto result = contacts::createContactsUser(providerWithoutCheckUserExists);
            if (!result) {
                return make_unexpected(result.error());
            }
        }

        return begin(providerWithCheckUserExists).bind(sync);
    }

    template <class ProviderWithCheckUserExists>
    expected<OrganizationRemovalResult> removeOrganizationImpl(
            TaskContextPtr context,
            ProviderWithCheckUserExists providerWithCheckUserExists) const {
        using services::db::begin;
        using services::db::request;
        using services::db::commit;
        using logic::UpdatedContacts;

        namespace contacts = logic::db::contacts;

        static_assert(services::db::ConnectionProvider<ProviderWithCheckUserExists>);

        const auto remove{[&](auto&& transaction) {

            auto unsubscribeFromContacts{[&](auto&&){

                auto orgListId{contacts::getDefaultListIdNoThrow(transaction)};
                if (!orgListId) {
                    return make_expected_from_error<OrganizationRemovalResult>(std::move(orgListId).error());
                }

                if (!orgListId->has_value()) {
                    return make_expected(OrganizationRemovalResult::Success);
                }

                auto sharedLists{contacts::getSharedLists(transaction)};
                if (!sharedLists) {
                    return make_expected_from_error<OrganizationRemovalResult>(std::move(sharedLists).error());
                }

                const auto filter{[&](const auto& list){return list.list_id == orgListId->value();}};
                const auto listTransformer{[&](const auto& list){return list.client_user_id;}};
                std::vector<std::int64_t> existingUserIds;
                boost::copy(*sharedLists | boost::adaptors::filtered(std::move(filter)) | boost::adaptors::
                        transformed(std::move(listTransformer)), std::back_inserter(existingUserIds));
                if (!std::is_sorted(existingUserIds.begin(), existingUserIds.end())) {
                    std::sort(existingUserIds.begin(), existingUserIds.end());
                }

                int failedUsersCount = 0;
                for (auto id : existingUserIds) {
                    auto userProvider = makeConnProviderWithCheckUserExists(context, PassportUserId{id});
                    const auto result = contacts::unsubscribeFromContacts(
                        transaction, std::move(userProvider), orgListId->value());
                    if (!result) {
                        failedUsersCount++;
                    }
                }

                if (failedUsersCount) {
                    LOGDOG_(context->logger(), error, log::user_id = providerWithCheckUserExists.uid(),
                        log::message="Failed to unsubscribe " + std::to_string(failedUsersCount) + " user when deleting organization");
                    return expected(OrganizationRemovalResult::Failure);
                }

                return expected(OrganizationRemovalResult::Success);
            }};

            const auto removeOrganization{[&](auto&& result){
                if (result == OrganizationRemovalResult::Success) {
                    const auto force = true;
                    const auto notFull = false;
                    contacts::purgeContactsUser(transaction, force, notFull);
                }

                return expected(result);
            }};

            const auto commit{[&](auto&& result){
                return services::db::commit(providerWithCheckUserExists,
                    std::move(transaction)).bind([&](auto &&){return expected(result);});}};

            return contacts::acquireRevision(transaction).
                bind(unsubscribeFromContacts).
                bind(removeOrganization).
                bind(commit).
                bind([&](auto&& result){return expected(result);});
        }};

        auto connection = getConnection(providerWithCheckUserExists);
        if (!connection) {
            auto ec = connection.error();
            if (ec == logic::Error::userNotFound) {
                return expected(OrganizationRemovalResult::Success);
            }
            return make_unexpected(ec);
        }

        return begin(providerWithCheckUserExists).bind(remove);
    }

    template<typename Handler> void manageSubscriptions(const TaskContextPtr& context,
            std::vector<std::int64_t> ids, Handler handler) const {
        for (auto id : ids) {
            auto result{handler(makeConnProviderWithCheckUserExists(context, PassportUserId{id}))};
            if (!result) {
                LOGDOG_(context->logger(), error, log::error_code=std::move(result).error(),
                        log::user_id=id, log::message="Handler error while subscription managing");
            }
        }
    }

    static void logError(TaskContextPtr context, error_code ec) {
        LOGDOG_(context->logger(), error, log::error_code=std::move(ec),
                log::message="organization has not been synced");
        boost::asio::steady_timer timer(getIoContext(context->yield()));
        timer.expires_after(std::chrono::seconds(1));
        timer.async_wait(context->yield());
    }

    MakeEventsQueueConnectionProvider makeEventsQueueConnectionProvider;
    MakeConnectionProviderWithCheckUserExists makeConnProviderWithCheckUserExists;
    MakeConnectionProviderWithoutCheckUserExists makeConnProviderWithoutCheckUserExists;
    DirectoryClientPtr directory;
    const Config& config;
};

} // namespace collie::directory_sync
