#include "workers_pool.hpp"
#include "run_worker.hpp"

#include <src/services/db/contacts/make_connection_provider.hpp>
#include <src/services/db/events_queue/make_connection_provider.hpp>
#include <src/services/directory/directory_client_impl.hpp>
#include <src/services/utils.hpp>

namespace collie::directory_sync {

WorkersPool::WorkersPool(boost::asio::io_context& io, const Config& config)
        : guard(boost::asio::make_work_guard(io)) {
    using MakeEventsQueueConnectionProvider = services::db::events_queue::MakeConnectionProvider;
    using DirectoryClient = services::directory::DirectoryClientImpl;
    using WithCheckIsUserExists = services::db::contacts::WithCheckIsUserExists;
    using WithoutCheckIsUserExists = services::db::contacts::WithoutCheckIsUserExists;
    using MakeConnectionProviderWithCheckUserExists = services::db::contacts::MakeConnectionProvider<WithCheckIsUserExists>;
    using MakeConnectionProviderWithoutCheckUserExists = services::db::contacts::MakeConnectionProvider<WithoutCheckIsUserExists>;


    const auto getHttpClient = services::makeGetHttpClient(config.services.httpClientModule);
    const auto getTvm2Module = services::makeGetTvm2Module(config.tvmModule);

    const RunWorker runWorker(
            MakeEventsQueueConnectionProvider(config.services.db.eventsQueue, getHttpClient),
            MakeConnectionProviderWithCheckUserExists(config.services.db.contacts, getHttpClient),
            MakeConnectionProviderWithoutCheckUserExists(config.services.db.contacts, getHttpClient),
            std::make_shared<DirectoryClient>(config.services.directory, getHttpClient, getTvm2Module),
            contexts,
            config);

    boost::for_each(boost::irange<std::size_t>(0, config.workersCount), [&] (auto index) {
        boost::asio::spawn(io, [=] (boost::asio::yield_context yield) mutable {
            runWorker(yield, index);
        }, boost::coroutines::attributes(config.coroutineStackSize));
    });
}

WorkersPool::~WorkersPool() {
    boost::for_each(*contexts.unique_lock(), [] (const auto& v) { v->stop(); });
    guard.reset();
}

} // namespace collie::directory_sync
