#pragma once

#include <mutex>
#include <shared_mutex>

namespace collie {

template <class T, class Mutex>
class unique_locked {
public:
    unique_locked(Mutex& mutex, T& value)
            : lock(mutex), value(value) {}

    decltype(auto) get() const {
        return value.get();
    }

    decltype(auto) operator ->() const {
        return &get();
    }

    decltype(auto) operator *() const {
        return get();
    }

private:
    std::unique_lock<Mutex> lock;
    std::reference_wrapper<T> value;
};

template <class T, class Mutex>
class shared_locked {
public:
    shared_locked(Mutex& mutex, const T& value)
            : lock(mutex), value(value) {}

    decltype(auto) get() const {
        return value.get();
    }

    decltype(auto) operator ->() const {
        return &get();
    }

    decltype(auto) operator *() const {
        return get();
    }

private:
    std::shared_lock<Mutex> lock;
    std::reference_wrapper<const T> value;
};

template <class T, class Mutex = std::mutex>
class guarded {
public:
    guarded() = default;

    guarded(const T& value)
            : value(value) {}

    guarded(T&& value)
            : value(std::move(value)) {}

    template <class ... Args>
    guarded(Args&& ... args)
            : value(std::forward<Args>(args) ...) {}

    template <class OtherMutex>
    guarded(const guarded<T, OtherMutex>& other)
            : value(other.shared_lock().get()) {}

    template <class OtherMutex>
    guarded(guarded<T, OtherMutex>&& other)
            : value(std::move(other.unique_lock().get())) {}

    auto unique_lock() {
        return unique_locked<T, Mutex>(mutex, value);
    }

    auto shared_lock() {
        return shared_locked<T, Mutex>(mutex, value);
    }

private:
    Mutex mutex {};
    T value;
};

} // namespace collie
