#pragma once

#include "contacts/create_contacts.hpp"
#include "settings_client_ptr.hpp"

#include <src/logic/interface/add_emails.hpp>
#include <src/services/db/begin.hpp>
#include <src/services/db/commit.hpp>
#include <src/services/db/get_connection.hpp>
#include <src/services/db/passport_user_id.hpp>
#include <src/services/recognizer/recognizer.hpp>

#include <butil/email/helpers.h>
#include <butil/StrUtils/StrUtils.h>
#include <butil/network/rfc2822.h>
#include <butil/network/idn.h>

#include <boost/algorithm/string.hpp>
#include <boost/range/algorithm_ext/erase.hpp>
#include <boost/range/join.hpp>

#ifdef __clang__
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wsign-conversion"
#endif
#include <mimeparser/rfc2047.h>
#ifdef __clang__
#pragma clang diagnostic pop
#endif

namespace collie::logic::db {

template <class MakeConnectionProvider>
class AddEmailsImpl final : public AddEmails {
public:
    explicit AddEmailsImpl(
        MakeConnectionProvider makeConnectionProvider,
        SettingsClientPtr settingsClient,
        const services::recognizer::RecognizerPtr& recognizer)
        : makeConnectionProvider(std::move(makeConnectionProvider))
        , settingsClient(std::move(settingsClient))
        , recognizer(recognizer) {}

    virtual expected<CreatedContacts> operator()(const TaskContextPtr& context, const Uid& uid,
            Recipients recipients) const override {
        std::int64_t convertedUid{0};
        if (!boost::conversion::try_lexical_convert(uid, convertedUid)) {
            LOGDOG_(context->logger(), error, log::uid=uid, log::message="failed to convert UID");
            return make_unexpected(error_code{Error::userNotFound});
        }

        const auto provider{makeConnectionProvider(context, services::db::PassportUserId{convertedUid})};
        auto result = getConnection(provider);
        if (!result) {
            return make_unexpected(result.error());
        }

        return settingsClient->getCollectAddressesField(context, uid)
            .bind([&](auto&& result) -> expected<CreatedContacts> {
                if (result) {
                    return processAddEmails(provider, context, convertedUid, std::move(recipients));
                }
                LOGDOG_(context->logger(), warning, log::uid=uid, log::message="collect addresses setting disabled");
                return CreatedContacts{};
            });
    }

    template<typename Connection>
    expected<CreatedContacts> processAddEmails(Connection& connection, const TaskContextPtr& context, std::int64_t uid,
            Recipients recipients) const {
        using namespace std::string_literals;

        const auto to{std::move(recipients.to).value_or(std::vector<std::string>{})};
        const auto cc{std::move(recipients.cc).value_or(std::vector<std::string>{})};
        const auto bcc{std::move(recipients.bcc).value_or(std::vector<std::string>{})};

        ::EmailVec validEmails;
        std::vector<std::string> invalidEmails;
        for (auto& email : boost::range::join(to, boost::range::join(cc, bcc))) {
            auto parsedEmail = EmailHelpers::fromString(email, std::nothrow_t{});
            if (parsedEmail.local().size() && parsedEmail.domain().size()) {
                validEmails.push_back(std::move(parsedEmail));
            } else {
                invalidEmails.push_back(std::move(email));
            }
        }

        if (invalidEmails.size()) {
            LOGDOG_(
                context->logger(),
                warning,
                log::uid=std::to_string(uid),
                log::message="invalid emails from request: "s + boost::algorithm::join(invalidEmails, ","));
        }

        if (validEmails.empty()) {
            LOGDOG_(context->logger(), warning, log::uid=std::to_string(uid), log::message="email list is empty");
            return CreatedContacts{};
        }

        return processRecipients(connection, context, std::move(validEmails));
    }

private:
    template<typename Connection>
    expected<std::vector<::Email>> getOnlyNewEmails(Connection& connection,
            std::vector<::Email> emails) const {
        services::db::contacts::query::GetOnlyNewEmails query;
        using services::db::unwrap;
        query.uid = unwrap(connection).uid();
        query.user_type = unwrap(connection).userType();
        std::transform(emails.begin(), emails.end(), std::back_inserter(query.emails),
                [](const auto& email){return email.addressString();});

        return services::db::request(connection, std::as_const(query)).bind([&](auto&& rows) {
            LOGDOG_(unwrap(connection).context()->logger(), notice, log::query=ozo::get_query_name(query),
                    log::uid=std::to_string(unwrap(connection).uid()));
            boost::remove_erase_if(emails, [&](const auto& email){
                    return std::count(rows.begin(), rows.end(), email.addressString()) == 0;});
            return emails;
        });
    }

    std::string trimRemoveCommasAndQuotes(std::string displayName) const {
        const auto searchedString{","};
        const auto substituteString{" "};
        boost::algorithm::replace_all(displayName, searchedString, substituteString);
        boost::algorithm::trim(displayName);
        if (displayName.size() >= 2) {
            const auto first{*displayName.cbegin()};
            if ((first == *displayName.crbegin()) && ((first == '\'') || (first == '\"'))) {
                const auto count{1};
                boost::algorithm::erase_head(displayName, count);
                boost::algorithm::erase_tail(displayName, count);
            }
        }
        return displayName;
    }

    std::string normalizeName(const std::string& name) const {
        std::string charset;
        std::string res = mulca_mime::decode_rfc2047(name, charset);
        utfizeString(*recognizer, res, charset);
        res = TStrUtils::removeSurroundQuotes(TStrUtils::unescape(res));
        return res;
    }

    logic::Name makeName(std::string rawDisplayName) const {
        auto normalizedDisplayName {normalizeName(std::move(rawDisplayName))};
        auto displayName{trimRemoveCommasAndQuotes(std::move(normalizedDisplayName))};
        std::vector<std::string> tokens;
        boost::algorithm::split(tokens, displayName, boost::algorithm::is_space(),
                boost::algorithm::token_compress_on);
        logic::Name name;
        if (tokens.size() == 2) {
            name.first = std::move(tokens[0]);
            name.last = std::move(tokens[1]);
        } else {
            name.first = std::move(displayName);
        }

        return name;
    }

    logic::Email makeEmail(::Email rawEmail) const {
        logic::Email email;
        auto domain = idna::decode(rawEmail.domain());
        email.email = rfc2822ns::join_address(rawEmail.local(), domain);;
        return email;
    }

    std::vector<NewContact> prepareContactsImpl(
            const TaskContextPtr& context,
            const std::vector<::Email>& newEmails) const {
        std::vector<NewContact> newContacts;
        std::transform(newEmails.begin(), newEmails.end(), std::back_inserter(newContacts),
                [&](const auto& newEmail) {
            NewContact newContact;
            try {
                const auto& displayName = newEmail.displayName();
                if (displayName.size() && displayName.find('@') == std::string::npos) {
                    newContact.vcard.names = {makeName(newEmail.displayName())};
                }

                newContact.vcard.emails = {makeEmail(std::move(newEmail))};
            } catch(const std::exception& e) {
                logException(context->logger(), e);
            }
            return newContact;
        });

        return newContacts;
    }

    template<typename Connection>
    expected<CreatedContacts> processRecipients(Connection& connection, const TaskContextPtr& context,
            std::vector<::Email> emails) const {
        const auto transact{[&](auto&& transaction) {
            const auto prepareContacts{[&](const auto& newEmails) {
                return prepareContactsImpl(context, newEmails);
            }};

            const auto createContacts{[&](auto&& newContacts) {
                CreatedContacts createdContacts;
                if (newContacts.size()) {
                    auto result = contacts::createContacts(std::move(newContacts), transaction);
                    if (!result) {
                        return result;
                    }
                    createdContacts = std::move(result.value());
                }
                return services::db::commit(connection, std::move(transaction))
                    .bind([&](auto&&){return std::move(createdContacts);});
            }};

            return getOnlyNewEmails(transaction, std::move(emails)).bind(prepareContacts).bind(
                    createContacts);
        }};

        return services::db::begin(connection).bind(transact);
    }

    MakeConnectionProvider makeConnectionProvider;
    SettingsClientPtr settingsClient;
    services::recognizer::RecognizerPtr recognizer;
};

} // namespace collie::logic::db
