#pragma once

#include <src/logic/interface/carddav_delete.hpp>
#include <src/logic/interface/types/reflection/existing_tag.hpp>

#include <src/logic/db/contacts/remove_contacts_completely.hpp>
#include <src/logic/db/contacts/get_contacts_by_tag_name_and_tag_type_and_uris.hpp>

#include <src/services/db/begin.hpp>
#include <src/services/db/commit.hpp>

namespace collie::logic::db {

template <class MakeConnectionProvider>
class CarddavDeleteImpl final : public CarddavDelete {
public:
    explicit CarddavDeleteImpl(MakeConnectionProvider makeConnectionProvider)
        : makeConnectionProvider(std::move(makeConnectionProvider)) {
    }

    virtual expected<void> operator()(
            const TaskContextPtr& context,
            const Uid& uid,
            const std::string& uri
        ) const override {
        std::int64_t numericUid;
        if (!boost::conversion::try_lexical_convert<std::int64_t>(uid, numericUid)) {
            return make_unexpected(error_code(Error::userNotFound));
        }
        auto provider {makeConnectionProvider(context, services::db::PassportUserId{
            numericUid})};

        const auto transact{[&](auto&& transaction) {
            const auto removeContacts{[&](auto&& contacts) {
                if (contacts.empty()) {
                    using namespace std::string_literals;
                    LOGDOG_(context->logger(), notice, log::uid = uid,
                        log::message = "for uri "s + uri + " not found phone contacts"s);
                    return make_expected(Revision{});
                }

                std::vector<ContactId> contactIds;
                boost::transform(contacts, std::back_inserter(contactIds),
                    [](const auto& contact){return contact.contact_id;});

                return contacts::removeContactsCompletely(transaction, contactIds);
            }};

            const auto commit{[&](auto&&) {
                return services::db::commit(std::move(provider), std::move(transaction)).
                        bind([&](auto&&){return;});
            }};

            return contacts::getContactsByTagNameAndTagTypeAndUris(
                transaction,
                toString(TagType::system),
                "Phone",
                {uri}
            ).
            bind(std::move(removeContacts)).
            bind(std::move(commit));
        }};

        return services::db::begin(provider).bind(std::move(transact));
    }

private:
    MakeConnectionProvider makeConnectionProvider;
};

} // namespace collie::logic::db
