#pragma once

#include "sheltie_client_ptr.hpp"

#include <src/logic/interface/carddav_multiget.hpp>
#include <src/logic/interface/types/carddav_contact.hpp>
#include <src/logic/interface/types/carddav_vcard_transforms.hpp>
#include <src/logic/interface/types/reflection/carddav_multiget_result.hpp>
#include <src/logic/interface/types/reflection/existing_tag.hpp>

#include <src/logic/db/carddav_utils.hpp>
#include <src/logic/db/contacts/get_contacts_by_tag_name_and_tag_type_and_uris.hpp>

#include <yamail/data/deserialization/json_reader.h>

namespace collie::logic::db {

template <class MakeConnectionProvider>
class CarddavMultigetImpl final : public CarddavMultiget {
public:
    explicit CarddavMultigetImpl(
        MakeConnectionProvider makeConnectionProvider,
        SheltieClientPtr sheltieClientPtr
    ) : makeConnectionProvider(std::move(makeConnectionProvider)),
        sheltieClientPtr(std::move(sheltieClientPtr)) {
    }

    virtual expected<CarddavMultigetResult> operator()(const TaskContextPtr& context, const Uid& uid,
            std::vector<std::string> uris) const override {
        std::int64_t numericUid;

        if (!boost::conversion::try_lexical_convert<std::int64_t>(uid, numericUid)) {
            return make_unexpected(error_code(Error::userNotFound));
        }

        CarddavMultigetResult carddavMultigetResult;

        if(uris.empty()) {
            LOGDOG_(context->logger(), error, log::uid=uid, log::message="uris list is empty");
            return carddavMultigetResult;
        }

        auto provider {services::db::retry(makeConnectionProvider(context, services::db::PassportUserId{
            numericUid}))};

        const auto makeCarddavMultigetResult {[&] (auto&& contacts) {
            boost::transform(uris, std::back_inserter(carddavMultigetResult.contact), [&](auto&& uri) {
                collie::logic::CarddavContact carddavContact;
                carddavContact.status = 404;
                carddavContact.uri = std::move(uri);
                return carddavContact;
            });

            if (contacts.empty()) {
                return carddavMultigetResult;
            }

            collie::logic::MapUriVcardJson mapUriVcardJson;
            try {
                boost::transform(contacts,  std::inserter(mapUriVcardJson, mapUriVcardJson.end()),
                        [&](auto&& contact) {
                    using yamail::data::deserialization::fromJson;
                    return std::pair<std::string, Vcard> {getUri(contact), fromJson<Vcard>(contact.vcard)};
                });
            } catch (const std::exception& ex) {
                using namespace std::string_literals;
                throw std::runtime_error("failed to parse contacts("s + ex.what() + ")"s);
            }

            std::map<std::string, std::string> mapUriEtag;
            boost::transform(contacts,  std::inserter(mapUriEtag, mapUriEtag.end()), [&](auto&& contact) {
                using yamail::data::deserialization::fromJson;
                return std::pair<std::string, std::string> {getUri(contact), getEtag(contact)};
            });

            auto mapUriVcardRfc = sheltieClientPtr->toVcard(context, uid, mapUriVcardJson);

            boost::for_each(carddavMultigetResult.contact, [&](auto&& contact) {
                const auto itVcard = mapUriVcardRfc.find(contact.uri);
                const auto itEtag = mapUriEtag.find(contact.uri);
                if (itVcard != mapUriVcardRfc.end() && itEtag != mapUriEtag.end()) {
                    contact.status = 200;
                    contact.etag = std::move(itEtag->second);
                    contact.vcard = std::move(itVcard->second);
                }
                else if(itEtag != mapUriEtag.end() && itVcard == mapUriVcardRfc.end()) {
                    using namespace std::string_literals;
                    LOGDOG_(context->logger(), warning, log::uid=uid,
                        log::message="for uri "s + contact.uri + " not find rfc vcard in sheltie response"s);
                }
            });
            return carddavMultigetResult;
        }};

        return contacts::getContactsByTagNameAndTagTypeAndUris(
            std::move(provider),
            toString(TagType::system),
            "Phone",
            uris
        ).bind(makeCarddavMultigetResult);
    }

private:
    MakeConnectionProvider makeConnectionProvider;
    SheltieClientPtr sheltieClientPtr;

};

} // namespace collie::logic::db
