#pragma once

#include <src/logic/interface/carddav_propfind.hpp>
#include <src/logic/interface/types/reflection/existing_tag.hpp>
#include <src/logic/interface/types/reflection/carddav_propfind_result.hpp>
#include <src/logic/interface/types/reflection/vcard.hpp>

#include <src/logic/db/carddav_utils.hpp>
#include <src/logic/db/contacts/get_contacts_by_tag_name_and_tag_type_and_uris.hpp>

#include <yamail/data/deserialization/json_reader.h>

namespace collie::logic::db {

template <class MakeConnectionProvider>
class CarddavPropfindImpl final : public CarddavPropfind {
public:
    explicit CarddavPropfindImpl(MakeConnectionProvider makeConnectionProvider)
        : makeConnectionProvider(std::move(makeConnectionProvider)) {
    }

    virtual expected<CarddavPropfindResult> operator()(const TaskContextPtr& context, const Uid& uid) const override {
        std::int64_t numericUid;
        if (!boost::conversion::try_lexical_convert<std::int64_t>(uid, numericUid)) {
            return make_unexpected(error_code(Error::userNotFound));
        }
        auto provider {services::db::retry(makeConnectionProvider(context, services::db::PassportUserId{
            numericUid}))};

        const auto makeCarddavPropfindResult {[] (auto&& contacts) {
            CarddavPropfindResult carddavPropfindResult;
            try {
                boost::transform(contacts, std::back_inserter(carddavPropfindResult.contact), [](const auto& contact) {
                    return CarddavContactInfo {
                        getUri(contact),
                        getDisplayName(yamail::data::deserialization::fromJson<Vcard>(contact.vcard)),
                        getEtag(contact)
                    };
                });
            } catch (const std::exception& ex) {
                using namespace std::string_literals;
                throw std::runtime_error("failed to parse contacts("s + ex.what() + ")"s);
            }
            return carddavPropfindResult;
        }};

        return contacts::getContactsByTagNameAndTagTypeAndUris(
            std::move(provider),
            toString(TagType::system),
            "Phone",
            {}
        ).bind(std::move(makeCarddavPropfindResult));
    }

private:
    MakeConnectionProvider makeConnectionProvider;
};

} // namespace collie::logic::db
