#pragma once

#include "sheltie_client_ptr.hpp"

#include <src/logic/interface/carddav_put.hpp>
#include <src/logic/interface/types/new_contact.hpp>
#include <src/logic/interface/types/reflection/existing_tag.hpp>
#include <src/logic/interface/types/updated_contacts.hpp>

#include <src/logic/db/carddav_utils.hpp>
#include <src/logic/db/contacts/create_contacts.hpp>
#include <src/logic/db/contacts/get_contacts_by_tag_name_and_tag_type_and_uris.hpp>
#include <src/logic/db/contacts/get_tag_id_by_tag_name_and_tag_type.hpp>
#include <src/logic/db/contacts/update_contacts.hpp>

#include <src/services/db/begin.hpp>
#include <src/services/db/commit.hpp>

#include <yamail/data/deserialization/json_reader.h>
#include <yamail/data/deserialization/yajl.h>
#include <yamail/data/serialization/yajl.h>

namespace collie::logic::db {

template <class MakeConnectionProvider>
class CarddavPutImpl final : public CarddavPut {
public:
    explicit CarddavPutImpl(
        MakeConnectionProvider makeConnectionProvider,
        SheltieClientPtr sheltieClientPtr)
        : makeConnectionProvider(std::move(makeConnectionProvider)),
        sheltieClientPtr(std::move(sheltieClientPtr)) {
    }

    virtual expected<CarddavPutResult> operator()(const TaskContextPtr& context, const Uid& uid,
            const std::string& uri, const std::string& etag, std::string vcardAsRfc) const override {
        using namespace std::string_literals;

        std::int64_t numericUid;
        if (!boost::conversion::try_lexical_convert<std::int64_t>(uid, numericUid)) {
            return make_unexpected(error_code(Error::userNotFound));
        }

        auto provider {makeConnectionProvider(context, services::db::PassportUserId{
            numericUid})};

        const auto transact{[&](auto&& transaction) {
            Vcard vcard;

            auto vcardAsJson = sheltieClientPtr->fromVcard(context, uid, std::move(vcardAsRfc));

            try {
                using yamail::data::deserialization::fromJson;
                vcard = fromJson<Vcard>(vcardAsJson);
            } catch (const std::exception& ex) {
                throw std::runtime_error("failed to parse contacts("s + ex.what() + ")"s);
            }

            const auto carddavPutImpl{[&](auto&& contacts) {
                if (etag == "*") {
                    return createContacts(context, uid, uri,
                        transaction, std::move(contacts), std::move(vcard)
                    );
                } else {
                    return updateContacts(context, uid, uri,
                        etag, transaction, std::move(contacts), std::move(vcard)
                    );
                }
            }};

            const auto commit{[&](auto&& carddavPutResult) {
                return services::db::commit(std::move(provider), std::move(transaction)).
                    bind([&](auto&&){return carddavPutResult;});
            }};

            return contacts::getContactsByTagNameAndTagTypeAndUris(
                transaction,
                toString(TagType::system),
                "Phone",
                {uri}
            ).
            bind(carddavPutImpl).
            bind(std::move(commit));
        }};
        return services::db::begin(provider).bind(std::move(transact));
    }

private:
    template<typename Connection> expected<CarddavPutResult> updateContacts(
            const TaskContextPtr& context,
            const std::string& uid,
            const std::string& uri,
            const std::string& etag,
            Connection&& connection,
            std::vector<collie::services::db::contacts::CarddavContactRow>&& contacts,
            Vcard&& vcard
    ) const {
        using namespace std::string_literals;
        CarddavPutResult carddavPutResult;

        if (contacts.size() != 1) {
            carddavPutResult.status = 404;
            carddavPutResult.description = "uri not found"s;
            LOGDOG_(context->logger(), warning, log::uid = uid,
                log::message = "not find contact with uri "s + uri);
                return  make_expected(carddavPutResult);
        }

        const auto contactForUpdate = std::move(contacts[0]);
        const auto existingeEtag = getEtag(contactForUpdate);

        if (existingeEtag != etag) {
            carddavPutResult.status = 409;
            carddavPutResult.description = "etag mismatch, contact etag ("s + existingeEtag
                + " != "s + etag + ")"s;
            LOGDOG_(context->logger(), warning, log::uid = uid,
                log::message = "etag mismatch, contact etag ("s + existingeEtag
                    + " != "s + etag + ")"s + "with uri "s + uri);
        } else {
            UpdatedContact updatedContact;
            updatedContact.contact_id = contactForUpdate.contact_id;
            updatedContact.vcard = std::move(vcard);
            updatedContact.uri = uri;

            UpdatedContacts updatedContacts;
            updatedContacts.updated_contacts.push_back(std::move(updatedContact));

            using contacts::updateContacts;
            const auto result = updateContacts(std::move(updatedContacts), connection);

            if (!result) {
                carddavPutResult.status = 400;
                carddavPutResult.description = "not update contact"s;
                LOGDOG_(context->logger(), warning, log::uid = uid,
                    log::message = "not update contact with uri "s + uri);
            } else {
                carddavPutResult.etag = getEtag(
                    contactForUpdate.contact_id,
                    result.value()
                );
                carddavPutResult.status = 200;
            }
        }
        return carddavPutResult;
    }

    template<typename Connection> expected<CarddavPutResult> createContacts(
            const TaskContextPtr& context,
            const std::string& uid,
            const std::string& uri,
            Connection&& connection,
            std::vector<collie::services::db::contacts::CarddavContactRow>&& contacts,
            Vcard&& vcard
    ) const {
        using namespace std::string_literals;
        CarddavPutResult carddavPutResult;

        if (!contacts.empty()) {
            carddavPutResult.status = 409;
            carddavPutResult.description = uri + " alredy exists";
            LOGDOG_(context->logger(), warning, log::uid = uid,
                log::message = "failed create contact with uri "s + uri);
            return make_expected(carddavPutResult);
        }

        const auto createPhoneContact{[&](auto&& tagId) {
            NewContact newContact;
            newContact.vcard = std::move(vcard);
            newContact.uri = uri;
            newContact.tag_ids = {std::move(tagId)};

            using contacts::createContacts;
            const auto result = createContacts({std::move(newContact)}, connection);

            if (!result || result->contact_ids.size() != 1) {
                carddavPutResult.status = 400;
                carddavPutResult.description = "failed create contact";
                LOGDOG_(context->logger(), warning, log::uid = uid,
                    log::message = "failed create contact with uri "s + uri);
            } else {
                carddavPutResult.etag = getEtag(
                result.value().contact_ids[0],
                result.value().revision
                );
                carddavPutResult.status = 201;
            }
                return  make_expected(carddavPutResult);
        }};

        return contacts::getTagIdByTagNameAndTagType(
            connection,
            toString(TagType::system),
            "Phone"
        ).bind(std::move(createPhoneContact));
    }

    MakeConnectionProvider makeConnectionProvider;
    SheltieClientPtr sheltieClientPtr;
};

} // namespace collie::logic::db
