#pragma once

#include "get_default_list_id.hpp"

#include <src/logic/interface/types/created_contacts.hpp>
#include <src/logic/interface/types/created_emails.hpp>
#include <src/logic/interface/types/new_contact.hpp>

#include <src/services/db/contacts/query.hpp>
#include <src/services/db/request.hpp>
#include <src/services/db/utils.hpp>

#include <src/expected.hpp>
#include <src/task_context.hpp>

#include <boost/range/algorithm/find_if.hpp>

#include <vector>

namespace collie::logic::db::contacts {

std::vector<services::db::contacts::NewContact> createQueryContacts(
        std::vector<NewContact>& newContacts, const std::optional<ListId>& defaultListId);

template <class Connection>
expected<CreatedContacts> createContactsImpl(std::vector<NewContact>& newContacts,
        const std::optional<ListId>& defaultListId, Connection& connection) {
    services::db::contacts::query::CreateContacts query;
    query.uid = services::db::unwrap(connection).uid();
    query.user_type = services::db::unwrap(connection).userType();
    query.contacts = createQueryContacts(newContacts, defaultListId);
    const auto context(services::db::unwrap(connection).context());
    query.x_request_id = context->requestId();
    return services::db::request(connection, std::as_const(query))
            .bind([&] (auto&& row) {
                auto result(std::get<0>(services::db::expectSingleRow(std::move(row))));
                LOGDOG_(context->logger(),
                        notice,
                        log::query=ozo::get_query_name(query),
                        log::uid=std::to_string(services::db::unwrap(connection).uid()),
                        log::revision=result.revision);

                return CreatedContacts{std::move(result.contact_ids), result.revision};
            });
}

std::vector<services::db::contacts::NewContactsEmail> createQueryEmails(
        const std::vector<ContactId>& contactIds, std::vector<NewContact>& newContacts);

std::map<TagId, std::vector<EmailId>> makeTagsToEmailIds(
    const CreatedEmails& cretedEmails,
    const std::vector<ContactId>& contactIds,
    const std::vector<NewContact>& newContacts);

template <class Connection>
expected<std::optional<Revision>> createAndTagContactsEmailsImpl(const std::vector<ContactId>& contactIds,
        std::vector<NewContact>& newContacts, Connection& connection) {
    services::db::contacts::query::CreateContactsEmails createEmailsQuery;
    auto newQueryEmails = createQueryEmails(contactIds, newContacts);
    createEmailsQuery.emails = newQueryEmails;
    if (createEmailsQuery.emails.empty()) {
        return std::optional<Revision>{};
    }

    createEmailsQuery.uid = services::db::unwrap(connection).uid();
    createEmailsQuery.user_type = services::db::unwrap(connection).userType();
    const auto context(services::db::unwrap(connection).context());
    createEmailsQuery.x_request_id = context->requestId();
    const auto result(services::db::request(connection, std::as_const(createEmailsQuery))
        .bind([&] (auto&& row) {
            auto result(std::get<0>(services::db::expectSingleRow(std::move(row))));
            LOGDOG_(context->logger(),
                    notice,
                    log::query=ozo::get_query_name(createEmailsQuery),
                    log::uid=std::to_string(services::db::unwrap(connection).uid()),
                    log::revision=result.revision);

            return CreatedEmails{std::move(result.email_ids), result.revision};
        }));
    if (!result) {
        return result.get_unexpected();
    }

    auto createdEmails = result.value();
    auto tagsToEmailIds = makeTagsToEmailIds(createdEmails, contactIds, newContacts);
    services::db::contacts::query::TagContactsEmails tagEmailsQuery;
    tagEmailsQuery.uid = services::db::unwrap(connection).uid();
    tagEmailsQuery.user_type = services::db::unwrap(connection).userType();
    tagEmailsQuery.x_request_id = context->requestId();
    std::optional<Revision> revision;
    for (auto& [tag_id, email_ids] : tagsToEmailIds) {
        tagEmailsQuery.tag_id = tag_id;
        tagEmailsQuery.email_ids  = std::move(email_ids);
        const auto result(services::db::request(connection, std::as_const(tagEmailsQuery))
            .bind([&] (auto&& row) {
                const auto revision(services::db::expectSingleRow(std::move(row)));
                LOGDOG_(context->logger(),
                        notice,
                        log::query=ozo::get_query_name(tagEmailsQuery),
                        log::uid=std::to_string(services::db::unwrap(connection).uid()),
                        log::revision=revision);

                return std::make_optional(revision);
            }));

        if (!result) {
            return result;
        }

        revision = result.value();
    }
    return revision;
}

std::unordered_map<TagId, std::vector<ContactId>> createTagToContactsMap(
        const std::vector<ContactId>& contactIds, const std::vector<NewContact>& newContacts);

template <class Connection>
expected<std::optional<Revision>> tagContactsImpl(const std::vector<ContactId>& contactIds,
        const std::vector<NewContact>& newContacts, Connection& connection) {
    auto tagToContacts(createTagToContactsMap(contactIds, newContacts));
    if (tagToContacts.empty()) {
        return std::optional<Revision>{};
    }

    services::db::contacts::query::TagContacts query;
    query.uid = services::db::unwrap(connection).uid();
    query.user_type = services::db::unwrap(connection).userType();
    const auto context(services::db::unwrap(connection).context());
    query.x_request_id = context->requestId();
    std::optional<Revision> revision;
    for (auto& [tag_id, contact_ids] : tagToContacts) {
        query.tag_id = tag_id;
        query.contact_ids  = std::move(contact_ids);
        const auto result(services::db::request(connection, std::as_const(query))
                .bind([&] (auto&& row) {
                    const auto revision(services::db::expectSingleRow(std::move(row)));
                    LOGDOG_(context->logger(),
                            notice,
                            log::query=ozo::get_query_name(query),
                            log::uid=std::to_string(services::db::unwrap(connection).uid()),
                            log::revision=revision);

                    return std::make_optional(revision);
                }));

        if (!result) {
            return result;
        }

        revision = result.value();
    }

    return revision;
}

template <class Connection>
expected<CreatedContacts> createContacts(std::vector<NewContact> newContacts, Connection&& connection) {
    const auto createContacts = [&] (const auto& defaultListId) {
        return createContactsImpl(newContacts, defaultListId, connection);
    };

    const auto createContactsEmailsAndTagContactsAndEmails = [&] (auto&& createdContacts) {
        const auto createAndTagEmails = [&] (auto&&) {
            return createAndTagContactsEmailsImpl(createdContacts.contact_ids, newContacts, connection);
        };

        const auto setRevision = [&] (std::optional<Revision>&& revision) {
            if (revision) {
                createdContacts.revision = *revision;
            }
            return std::move(createdContacts);
        };

        return tagContactsImpl(createdContacts.contact_ids, newContacts, connection)
            .bind(createAndTagEmails)
            .bind(setRevision);
    };

    expected<std::optional<ListId>> defaultListId;

    const auto hasListId = [&] (const auto& v) { return v.list_id.has_value(); };
    if (boost::find_if(newContacts, std::not_fn(hasListId)) != newContacts.end()) {
        defaultListId = getDefaultListId(connection)
            .bind([&] (auto v) { return expected(std::optional(v)); });
    }

    return defaultListId
        .bind(createContacts)
        .bind(createContactsEmailsAndTagContactsAndEmails);
}

template <class Connection>
expected<CreatedContacts> createContacts(std::vector<NewContact> newContacts, Connection&& connection, size_t chunkSize) {
    std::vector<NewContact> chunk;
    CreatedContacts createdContacts;
    for (std::size_t index{0}; index < newContacts.size(); ++index) {
        chunk.emplace_back(std::move(newContacts[index]));
        if ((chunk.size() == chunkSize) || (index == (newContacts.size() - 1))) {
            auto result = createContacts(chunk, connection);
            if (!result) {
                return result;
            }
            auto& createdContactsChunk = result.value();
            createdContacts.revision = createdContactsChunk.revision;
            createdContacts.contact_ids.insert(
                createdContacts.contact_ids.end(),
                createdContactsChunk.contact_ids.begin(),
                createdContactsChunk.contact_ids.end());
            chunk.clear();
        }
    }
    return createdContacts;
}

template<class Connection> expected<std::vector<ContactId>> createContactsAndReturnIds(
        std::vector<NewContact> newContacts, Connection&& connection, size_t chunkSize) {
    auto getContactIds{[](auto&& createdContacts){return std::move(createdContacts.contact_ids);}};
    return createContacts(std::move(newContacts), connection, chunkSize).bind(std::move(getContactIds));
}

} // namespace collie::logic::db::contacts
