#pragma once

#include <src/expected.hpp>
#include <src/logic/interface/types/created_emails.hpp>
#include <src/services/db/contacts/query.hpp>
#include <src/services/db/request.hpp>
#include <src/services/db/utils.hpp>

namespace collie::logic::db::contacts {

template<typename Connection> expected<CreatedEmails> createContactsEmailsImpl(Connection&& connection,
        std::vector<services::db::contacts::NewContactsEmail> emails) {
    services::db::contacts::query::CreateContactsEmails query;
    const auto uid{services::db::unwrap(connection).uid()};
    query.uid = uid;
    query.user_type = services::db::unwrap(connection).userType();
    query.emails = std::move(emails);
    const auto context{services::db::unwrap(connection).context()};
    query.x_request_id = context->requestId();
    return services::db::request(connection, std::as_const(query)).bind([&](auto&& rows) {
        const auto result{std::get<0>(services::db::expectSingleRow(std::move(rows)))};
        LOGDOG_(context->logger(), notice, log::query=ozo::get_query_name(query),
                log::uid=std::to_string(uid), log::revision=result.revision);
        return CreatedEmails{std::move(result.email_ids), result.revision};
    });
}

template<typename Connection> expected<CreatedEmails> createContactsEmailsImpl(Connection&& connection,
        std::vector<services::db::contacts::NewContactsEmail> emails, size_t chunkSize) {
    auto begin{emails.begin()};
    CreatedEmails result;
    while (begin != emails.end()) {
        auto end{(static_cast<size_t>(std::distance(begin, emails.end())) >= chunkSize) ? begin + chunkSize :
                emails.end()};
        auto createdEmails{createContactsEmailsImpl(connection, {std::make_move_iterator(begin),
                std::make_move_iterator(end)})};
        if (!createdEmails) {
            return createdEmails;
        }

        begin = end;
        result.email_ids.insert(result.email_ids.end(), createdEmails->email_ids.begin(),
                createdEmails->email_ids.end());
        result.revision = createdEmails->revision;
    }

    return result;
}

template<typename Connection> expected<CreatedEmails> createContactsEmails(
        Connection&& connection,
        std::vector<services::db::contacts::NewContactsEmail> emails,
        const std::optional<size_t>& chunkSize) {
    if (chunkSize) {
        return createContactsEmailsImpl(connection, std::move(emails), *chunkSize);
    } else {
        return createContactsEmailsImpl(connection, std::move(emails));
    }
}

}
