#pragma once

#include "get_contacts_email_ids_by_contacts_ids.hpp"
#include "remove_contacts.hpp"
#include "remove_contacts_emails.hpp"
#include "untag_contacts_completely.hpp"
#include "untag_contacts_emails_completely.hpp"


#include <src/expected.hpp>
#include <src/services/db/contacts/query.hpp>
#include <src/services/db/request.hpp>

namespace collie::logic::db::contacts {

template<typename Connection> expected<Revision>
removeContactsCompletely(Connection&& connection, const std::vector<ContactId>& contactIds) {
    const auto removeContactsEmails{[&](auto&& emailIdRows) {
        if (emailIdRows.empty()) {
            return make_expected(Revision{});
        }
        std::vector<std::int64_t> emailIds;
        std::transform(emailIdRows.begin(), emailIdRows.end(), std::back_inserter(emailIds),
            [](const auto& emailIdRow){return emailIdRow.email_id;});

        const auto revision = contacts::untagContactsEmailsCompletely(connection, emailIds);

        if (!revision) {
            return revision;
        }
        return contacts::removeContactsEmails(connection, std::move(emailIds));
    }};

    const auto removeContacts{[&](auto&&) {
        const auto revision = contacts::untagContactsCompletely(connection, contactIds);

        if(!revision) {
            return revision;
        }
        return contacts::removeContacts(connection, contactIds);
    }};

    return contacts::getContactsEmailIdsByContactsIds(connection, contactIds).
            bind(std::move(removeContactsEmails)).
            bind(std::move(removeContacts));
}

template <class Connection>
expected<Revision> removeContactsCompletely(Connection&& connection, const std::vector<ContactId>& contactIds, size_t chunkSize) {
    auto result = expected<Revision>();
    std::vector<ContactId> chunk;
    for (std::size_t index{0}; index < contactIds.size(); ++index) {
        chunk.emplace_back(std::move(contactIds[index]));
        if ((chunk.size() == chunkSize) || (index == (contactIds.size() - 1))) {
            result = removeContactsCompletely(connection, chunk);
            if (!result) {
                return result;
            }
            chunk.clear();
        }
    }
    return result;
}

} // collie::logic::db::contacts
